/*                        PhyloGibbs                                  */

/*   Algorithm developed by Rahul Siddharthan, Erik van Nimwegen      * 
 *   and Eric D. Siggia at The Rockefeller University, New York       * 
 *                                                                    *
 *   This code copyright (C) 2004 Rahul Siddharthan <rsidd@online.fr> * 
 *   Licensed under the GNU General Public License (see COPYING)      */ 

/* 
 * $Author: rsidd $  
 * $Date: 2005/05/02 08:54:09 $ 
 * $Id: gibbscolourmove.c,v 1.1 2005/05/02 08:54:09 rsidd Exp $ 
 */

#include "stdio.h"
#include <math.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_sf_gamma.h>

#include "interspecies.h"
#include "moveweight.h"
#include "fullbinweight.h"
#include "binbasecount.h"
#include "windowscore.h"
#include "recolourwindow.h"


/*  This does the following gibbs move:
 *
 *  o  Pick one window, either at random or sample each possible one
 *     sequentially.
 *
 *  o  If blocked: no move, increase time counter by 1
 *
 *  o  If coloured: "temporarily" blank it white (zero) and also blank
 *     its pair window.  If white already go on to next step.
 *
 *  o  Sample from possible colours for this window and all possible
 *     pair windows, could be white, existing colour, or new colour.  
 *     Note - if earlier it was the only window in its colour, "new 
 *     colour" really means "same colour as earlier".
 *
 *  o  If a window is of colour N (odd), its pair is to the right (if
 *     not RC) or left (if RC), and of colour N+1.  If one window in
 *     pair is RC, so is the other.  Allowed pairs are listed in the 
 *     window struct as the GArrays lpair and rpair.
 *
 */

int gibbscolourmove(params *v, double *logprevweight) 
{
    int ncol,nwin,nc,nm,ndir,numnewcol,n;
    onemove trialmove,*thismove; 
    GArray *moves;
    window *currwin;
    double totalweight,moverandom,currweight,oldprevweight,truefullbinweight,
        maxweight,newcolweight;


    if (v->usedir!=0) 
        v->usedir=1;
    oldprevweight=*logprevweight;


    maxweight = -100000.0;

    do {
        nwin=gsl_rng_uniform_int(v->gslrand, (v->win)->len);
        currwin=&g_array_index((v->win),window,nwin);
    } while ((currwin->permblocked));

    if ((currwin->blocked))
        return 1;

    moves=g_array_new(TRUE,TRUE,sizeof(onemove));
     
    if (currwin->colour>0)
        blankwindow(v,currwin);
    
    /**Score with window blanked***/
    makebinbasecount(v);
    *logprevweight=fullbinweight(v);

    /**WM file. See if we can have new color and what the weight is ****/
    if((v->priorbinbase)->len > 0)
      {
        numnewcol = 0;
	n = ((v->priorbinbase)->len)+1;
	while(n<((v->bin)->len))
	  {
	    if(bincount(v->bin,n) > 0)
	      ++numnewcol;
	    ++n;
	  }
        newcolweight = (double) (v->numtfs - numnewcol);
        if(newcolweight > 0)
          {
            ncol = (v->bin)->len+1;
            newcolweight = log(newcolweight);
	    newcolweight = newcolweight/(v->beta);
          }
        else
          {
            ncol = (v->bin)->len;
            newcolweight = 0.0;
          }
      }
    /***no WM file. Always allow new color****/
    else
      {
        ncol=(v->bin)->len+1;/**0 plus colors + 1***/
        newcolweight = 0.0;
      }

    if(ncol < ((v->priorbinbase)->len +2))
      {
	fprintf(stderr,"Something wrong there should be at least %d colors but there are only %d\n",(v->priorbinbase)->len,(v->bin)->len);
      }
    
    for (nc=0; nc<ncol; nc++) {
        for (ndir=0; ndir<=v->usedir; ndir++) {
            trialmove.win=currwin;
            trialmove.dir=ndir;
            currwin->dir=ndir;
            trialmove.oldcolour=0;
            trialmove.newcolour=nc;
            trialmove.moveweight=moveweight(v,&trialmove);
	    /**nonzero colour, subtract window chem pot***/
            if (trialmove.newcolour>0) 
                trialmove.moveweight=trialmove.moveweight-(v->mu);
	    /**new colour, subtract colour chem pot***/
	    if(trialmove.newcolour>=((v->bin)->len))
	      {
		trialmove.moveweight -= (v->lambda);
		trialmove.moveweight += newcolweight;
	      }
	    /**currently empty reference bin, is also new color****/
            else if(bincount(v->bin,trialmove.newcolour) == 0)
              {
                trialmove.moveweight -= (v->lambda);
              }
            if (trialmove.moveweight > maxweight)
                maxweight = trialmove.moveweight;
            g_array_append_val(moves,trialmove);
	}
    }
    /* normalise the moveweights */
    totalweight = 0.0;
    for (nm=0;nm<(moves->len);++nm) {
        thismove = &g_array_index(moves,onemove,nm);
        thismove->moveweight = exp((v->beta)*(thismove->moveweight-maxweight));
        totalweight += thismove->moveweight;
    }
    
    /* now pick a move from moves, weighted by moveweight */
    moverandom=gsl_rng_uniform(v->gslrand)*totalweight;
    currweight=0.0;
    nm=0;
    do {
        trialmove=g_array_index(moves,onemove,nm);
        nm++;
        currweight=currweight+trialmove.moveweight;
    } while (currweight<moverandom);


    *logprevweight=*logprevweight+log(trialmove.moveweight)
        + (v->beta)*maxweight;

    if(trialmove.newcolour >= (v->bin)->len)
      {
        *logprevweight -= (v->beta)*newcolweight;
      }

    /* and update binset and windowset accordingly; if the "occupancy"
     * of a bin drops to zero, remove it from the binset and move the
     * other members "up" (updating also their colours)
     */ 
    if (trialmove.newcolour==0) {
        assert(fabs(log(trialmove.moveweight)+(v->beta)*maxweight) < 0.00001);
        currwin->colour=0;
        g_array_free(moves,TRUE);
        return 0;
    }
    
    colourwindow(v,&trialmove);
    
    g_array_free(moves,TRUE);


    /* check that prevweight is being kept track of properly */

    truefullbinweight=fullbinweight(v);
    if(fabs((*logprevweight)-truefullbinweight)  > 0.1)
      fprintf(stderr,"WARNING: abnormally large difference between the calculated score %lf and\n the true full score %lf in gibbswindowmove\n",*logprevweight,truefullbinweight);
    *logprevweight = truefullbinweight;



    /* we _could_ update binbase cleanly 
     * but don't bother -- this looks fast enough 
     */

    makebinbasecount(v);

    return 0;

}    
