/*                        PhyloGibbs                                  */

/*   Algorithm developed by Rahul Siddharthan, Erik van Nimwegen      * 
 *   and Eric D. Siggia at The Rockefeller University, New York       * 
 *                                                                    *
 *   This code copyright (C) 2004 Rahul Siddharthan <rsidd@online.fr> * 
 *   Licensed under the GNU General Public License (see COPYING)      */ 

/* 
 * $Author: rsidd $  
 * $Date: 2005/04/29 18:38:57 $ 
 * $Id: wminteg.c,v 1.4 2005/04/29 18:38:57 rsidd Exp $ 
 */

/* 2005/04/14 Erik van Nimwegen
 * Made changes to incorporate pseudocounts for the 
 * prior of the WM integral in such a way that the 
 * calculation is still fast for integer counts of
 * bases. 
 * It uses the general property of the Gamma function
 * that Gamma(x+1) = x Gamma(x)
 */


#include <glib.h>
#include <math.h>
#include <gsl/gsl_sf_gamma.h>
#include "wminteg.h"

/***contains log(gamma(n+pseudocount)/gamma(pseudocount))***/
static double loggamplusq[100000];
/***contains log(gamma(n+4*pseudocount)/gamma(4*pseudocount))****/
static double loggamplus4q[100000];

static double pc;
static double lgq;
static double lg4q;

/** initialize table with values of Log[Gamma(n+q)/Gamma(q)] ***
    and Log[Gamma[(n+4q)/Gamma(4q)]] ***************************/
void inittables(double pseudocount)
{
    int n;
    loggamplusq[0] = 0;
    loggamplus4q[0] = 0;
    pc = pseudocount;
    lgq = gsl_sf_lngamma(pc);
    lg4q = gsl_sf_lngamma(4.0*pc);
    for(n=1;n<100000;n++) {
      loggamplusq[n] = gsl_sf_lngamma(pc+0.01*((double) n))-lgq;
      loggamplus4q[n] = gsl_sf_lngamma(4.0*pc+0.01*((double) n))-lg4q;
    }
}

/**calculate the log-ratio log[Gamma(n1+q)/Gamma(q)]***/
inline double gam_ratio_q(double n1) {
    if(n1 >=0){
      if(n1 < 1000){
	double del = 100.0*n1;
	int n = ((int) del);
	del -= ((double) n);
	double res = (1.0-del)*loggamplusq[n];
	++n;
	res += del*loggamplusq[n];
	return (res);
      }
      else{
	double x = n1+pc;
	return ((x-0.5)*log(x)-x+0.918939+0.0833333/x -lgq);
      }
    }
    else{
      return (gsl_sf_lngamma(n1+pc)-lgq);
    }
}


/**calculate the log-ratio log[gamma(n+4q)/gamma(4q)] **/
inline double gam_ratio_4q(double n1) {
    if(n1>=0){
      if(n1 < 1000){
	double del = 100.0*n1;
        int n = ((int) del);
        del -= ((double) n);
        double res = (1.0-del)*loggamplus4q[n];
        ++n;
        res += del*loggamplus4q[n];
        return (res);
      }
      else{
	double x = n1+4.0*pc;
	return ((x-0.5)*log(x)-x+0.918939+0.0833333/x -lg4q);
      }
    }
    else{
      return (gsl_sf_lngamma(n1+4.0*pc)-lg4q);
    }
}

/**calculate log of Gamma[4q]/Gamma[n+4q] prod_a (Gamma[na+q]/Gamma[q])  ****/
double wminteg(double n1, double n2, double n3, double n4) {
    double integ = 0;
    integ += gam_ratio_q(n1);
    integ += gam_ratio_q(n2);
    integ += gam_ratio_q(n3);
    integ += gam_ratio_q(n4);
    integ -= gam_ratio_4q(n1+n2+n3+n4);
    return integ;
}

/** precise integral used during window initialization 
    (needs precise numbers for arguments between 0 and 1)***/
double precise_wminteg(double n1, double n2,double n3,double n4){
  double integ = 0;
  integ += (gsl_sf_lngamma(n1+pc)-lgq);
  integ += (gsl_sf_lngamma(n2+pc)-lgq);
  integ += (gsl_sf_lngamma(n3+pc)-lgq);
  integ += (gsl_sf_lngamma(n4+pc)-lgq);
  integ -= (gsl_sf_lngamma(n1+n2+n3+n4+4.0*pc)-lg4q);
  return integ;
}
