/*                        PhyloGibbs                                  */

/*   Algorithm developed by Rahul Siddharthan, Erik van Nimwegen      * 
 *   and Eric D. Siggia at The Rockefeller University, New York       * 
 *                                                                    *
 *   This code copyright (C) 2004 Rahul Siddharthan <rsidd@online.fr> * 
 *   Licensed under the GNU General Public License (see COPYING)      */ 

/* 
 * $Author: rsidd $  
 * $Date: 2005/05/22 13:29:54 $ 
 * $Id: parseopts.c,v 1.6 2005/05/22 13:29:54 rsidd Exp $ 
 */

#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include "interspecies.h"
#include "tree_routines.h"
#include "getoccstr.h"


int parseopts(int argc, char *argv[], params *v) 
{


    char ch,Nstr[256];
    int n,commalist;
    double bgtotal;

	static struct option longopts[] = {
			{ "trackfile",   required_argument,  0,  'A' },
			{ "blockedfile", required_argument,  0,  'B' },
			{ "rcsymmetric",  no_argument,        0,  'C' },
			{ "dialign",     required_argument,  0,  'D' },
            { "trackingcutoff", required_argument, 0, 'E' },
			{ "bgfile",      required_argument,  0,  'F' },
            { "phylohistory", required_argument,  0,  'G' },
			{ "phylohistlist", required_argument,  0,  'H' },
            { "initialocc",  required_argument,  0,  'I' },
            { "labeltree", required_argument, 0, 'L'},
			{ "ncorrel",     required_argument,  0,  'N' },
			{ "bgpseudocount",   required_argument,  0,  'P' },
            { "reverseprint", no_argument, 0, 'R' },
			{ "nsteps",     required_argument,  0,  'S' },
			{ "pseudocount", required_argument,  0,  'T' },
            { "write-each-cycle", no_argument, 0, 'W' },
			{ "noautotrack", no_argument,        0,  'X' },
			{ "nanneal", required_argument, 0, 'a' },
			{ "beta",        required_argument,  0,  'b' },
			{ "ncolmoves",   required_argument,  0,  'c' },
			{ "inputfile",   required_argument,  0,  'f' },
			{ "ndeepquench", required_argument, 0, 'g'},
			{ "help",        no_argument,        0,  'h' },
			{ "initfile",    required_argument,  0,  'i' },
			{ "motifwidth",  required_argument,  0,  'm' },
			{ "outputfile",  required_argument,  0,  'o' },
			{ "chempot",     required_argument,  0,  'p' },
			{ "quiet",       no_argument,        0,  'q' },
			{ "norevcomp",   no_argument,        0,  'r' },
			{ "nshiftmoves", required_argument,  0,  's' },
			{ "trackedoutput",   required_argument,  0,  't' },
			{ "ntransient", required_argument, 0, 'u' },
			{ "verbose", no_argument, 0, 'v' },
			{ "nwinmoves",   required_argument,  0,  'w' },
			{ "betaincr",   required_argument,  0,  'x' },
			{ "nexpwin", required_argument, 0, 'y'},
			{ "nexpcol", required_argument, 0, 'z'},
			{ "motiffile", required_argument, 0, 'M'},
			{0, 0, 0, 0}
	};		
    

    while ((ch = getopt_long(argc,argv,"A:B:CD:E:F:G:H:I:L:M:N:P:RS:T:WXa:b:c:f:g:hi:k:m:o:p:qrs:t:u:vw:x:y:z:", longopts, NULL)) != -1) {
        switch (ch) {
        case 'q': /* quiet output */
            v->quiet=1;
            v->verbose=0;
            break;
        case 'v': /* verbose output */
            v->verbose=1;
            v->quiet=0;
            break;
        case 'W': /* write output file every cycle */
            v->writeatend=0;
            break;
        case 'f': /* name of input sequence file */
            strcpy(v->seqfile,optarg);
            break;
        case 'F': /* name of bg file */
            strcpy(v->bgfile,optarg);
            break;
	case 'M': /** name of motif file */    
	  strcpy(v->motiffile,optarg);
	  break;
        case 'm': /* motif width */
            strcpy(Nstr,optarg);
            v->wwidth=atoi(Nstr);
            if (v->wwidth<=0) {
                fprintf(stderr,"Error: motif width (-m) must be positive integer!\n");
                return 1;
            }
            break;
        case 'X': /* don't automatically track annealed windows */
            v->autotrack=0;
            break;
        case 'r': /* don't search for reverse complement matches */
            v->usedir=0;
            break;
        case 'C': /* look for reverse-complement symmetric motifs */
            v->rcsymm=1;
            break;
        case 'D': /* use dialign */
            strcpy(Nstr,optarg);
            v->usedialign=atoi(Nstr);
            if ((v->usedialign<0)||(v->usedialign>2)) {
                
                fprintf(stderr,"Error: Legal values for -D are 0 (unaligned sequences), 1, 2\n");
                return 1;
            }
            break;
        case 'S': /* total number of iterations */
            strcpy(Nstr,optarg);
            if (atoi(Nstr)<0)
                v->totaliters=0;
            else
                v->totaliters=atoi(Nstr);
            break;
        case 'u': /*number of iterations in the transient**/
            strcpy(Nstr,optarg);
            if (atoi(Nstr)<0)
                v->transientiters=0;
            else
                v->transientiters=atoi(Nstr);
            break;	
        case 'a': /*number of iterations in the anneal**/
            strcpy(Nstr,optarg);
            if (atoi(Nstr)<0)
                v->annealiters=0;
            else
                v->annealiters=atoi(Nstr);
            break;
        case 'g': /*number of iterations in the deep quench**/
            strcpy(Nstr,optarg);
            if (atoi(Nstr)<0)
                v->deepquenchiters=0;
            else
                v->deepquenchiters=atoi(Nstr);
            break;
        case 'T': /* use traditional Wadsworth-style scoring */
            strcpy(Nstr,optarg);
            v->pseudocount = atof(Nstr);
            break;
        case 'R': /* print seq position as negative number from end */
            v->printreverse=1;
            break;
        case 'N': /* correlated background, -1= 1/4 each base,
                   * 0 = raw basecounts from seq, 1, 2... =
                   * conditional probs based on 1, 2, ..
                   * neighbours   
                   * if a comma-separated float list, set
                   * components to **givenbgcounts, and set nbgc
                   * to -2 */
            strcpy(Nstr,optarg);
            commalist=0;
            for (n=0; n<strlen(Nstr); n++) 
                if (Nstr[n]==',') {
                    commalist=1; 
                    break;
                }
            if (commalist) {
                v->nbgc=-2;
                getoccstr(Nstr,&(v->givenbgcounts),'f');
                if ((v->givenbgcounts)->len !=4) {
                    fprintf(stderr,"WARNING: ill-formed -N argument, assuming bg prob 0.25 per base\n");
                    g_array_free((v->givenbgcounts),TRUE);
                    v->givenbgcounts=NULL;
                    v->nbgc=-1;
                } 
                else {
                    bgtotal=0.0;
                    for (n=0; n<4; n++)
                        bgtotal=bgtotal+g_array_index((v->givenbgcounts),double,n);
                    if (bgtotal==0.0) {
                        g_array_free((v->givenbgcounts),TRUE);
                        v->givenbgcounts=NULL;
                        v->nbgc=-1;
                    }
                    else {
                        for (n=0; n<4; n++)
                            g_array_index((v->givenbgcounts),double,n)=
                                g_array_index((v->givenbgcounts),double,n)/bgtotal;
                    }
                }
            } 
            else {
                v->nbgc=atoi(Nstr);
                if (v->nbgc<-1) 
                    v->nbgc=-1;
            }
            break;
        case 'E': /* tracking cutoff */
            strcpy(Nstr,optarg);
            v->trackingcutoff=atof(Nstr);
            if ((v->trackingcutoff<0.0)||(v->trackingcutoff>=1.0)) {
                fprintf(stderr,"Warning: tracking cutoff (-E) must be between 0 and 1, assuming 0.05\n");
                v->trackingcutoff=0.05;
            }
            break;
        case 'b': /* value of beta */
            strcpy(Nstr,optarg);
            v->beta=atof(Nstr);
            if (v->beta<=0.0) {
                fprintf(stderr, "Error: beta (-b) must be positive!");
                return 1;
            }
            break;
        case 'x': /* value of beta increment */
            strcpy(Nstr,optarg);
            v->betaincr=atof(Nstr);
            if (v->betaincr < -0.000000001) { /* disallow negative values */
                fprintf(stderr,"Error: negative values not allowed for betaincr (-x); using automatic value\n");
                return 1;
            }
            break;
        case 'y': /*number of expected windows**/
	  strcpy(Nstr,optarg);
	  v->deswin = atoi(Nstr);
	  if(v->deswin <= 0){
	    fprintf(stderr,"Error: number of windows has to be bigger than zero\n");
	    return 1;
	  }
	  break;
	case 'z': /*number of expected colors**/
	  strcpy(Nstr,optarg);
	  v->descol = atoi(Nstr);
	  if(v->descol <= 0){
	    fprintf(stderr,"Error: number of colors has to be bigger than zero\n");
	    return 1;
	  }
	  break;
        case 'P': /* pseudocount of single-site correls */
            v->bgpscount=atof(Nstr);
            if (v->bgpscount < 0.0) {
                fprintf(stderr, "Error: background pseudocount (-P) must be positive!\n");
                return 1;
            }
            break;
        case 'p': /* value of chemical potential mu */
            strcpy(Nstr,optarg);
            v->mu=atof(Nstr);
            break;
        case 'c': /* number of single colour moves */
            strcpy(Nstr,optarg);
            v->Ncolm=atoi(Nstr);
            break;
        case 'w': /* number of window-shift moves */
            strcpy(Nstr,optarg);
            v->Nwinm=atoi(Nstr);
            break;
        case 's': /* number of global-shift moves */
            strcpy(Nstr,optarg);
            v->Nshiftm=atoi(Nstr);
            break;
	case 'i': /* name of initial position file*/
            strcpy(v->posfile,optarg);
            break;
        case 'B': /* name of blocked window file*/
            strcpy(v->blockedfile,optarg);
            break;
        case 'H': /* get seqmuset, ie phylogenetic history */
            strcpy(Nstr,optarg);
            getoccstr(Nstr,&(v->seqmuset),'f');
            for (n=0; n<v->seqmuset->len; n++)
                if (g_array_index(v->seqmuset,double,n)<0.0) {
                    fprintf(stderr,"Error: The phylogeny parameter (-G) must be positive!\n");
                    return 1;
                }
            break;
        case 'L': /* convert label tree string to tree struct */
            v->labeltree=g_node_new(NULL);
            stringtotree(optarg,&(v->labeltree),0);
            break;
	case 'G': /* get nomutprob, ie default value of phylogenetic
                     history */
            strcpy(Nstr,optarg);
            v->nomutprob=atof(Nstr);
            if (v->nomutprob < 0.0) {
                fprintf(stderr,"Error: The phylogeny parameter (-G) must be positive!\n");
                return 1;
            }
            break;
        case 'I': /* initial occupation set in format n1,n2,n3 */
            strcpy(Nstr,optarg);
            getoccstr(Nstr,&(v->occset),'i');
            break;
        case 'o': /* name of output file*/
            strcpy(v->outfile,optarg);
            break;
        case 't': /* name of tracking output file */
            strcpy(v->trackedprintfile,optarg);
            break;
        case 'A': /*name of tracked bin file */
            strcpy(v->trackedbinfile,optarg);
            break;
        case 'h': /* help */
            v->gethelp=1;
            break;
        default: /* get help, exit */
            v->gethelp=1;
        }
    }
    
    return 0;
}

