/*                        PhyloGibbs                                  */

/*   Algorithm developed by Rahul Siddharthan, Erik van Nimwegen      * 
 *   and Eric D. Siggia at The Rockefeller University, New York       * 
 *                                                                    *
 *   This code copyright (C) 2004 Rahul Siddharthan <rsidd@online.fr> * 
 *   Licensed under the GNU General Public License (see COPYING)      */ 

/* 
 * $Author: rsidd $  
 * $Date: 2005/05/02 08:54:09 $ 
 * $Id: gibbswindowmove.c,v 1.1 2005/05/02 08:54:09 rsidd Exp $ 
 */

#include <stdio.h>
#include <math.h>
#include <gsl/gsl_rng.h>
#include "interspecies.h"
#include "moveweight.h"
#include "fullbinweight.h"
#include "binbasecount.h"
#include "windowscore.h"
#include "recolourwindow.h"

/*  This function does the following
 *  1. Pick a window at random and blank it, and note the region it came from.
 *  2. If 'extra region' is set, a second  region is chosen at random.
 *  3. Calculate the score that would be obtained by coloring any of the unblocked windows
 *     in the region(s) with any of the allowed colors (current colors + 1 if new colors allowed)
 *  4. Sample a target window + color 
 */

int switchwindow(params *v, double *logprevweight,int extraregion) 
{
    int nbin,nwins,n,nc,nb,nm,ndir,thiswinnum,zer,
        oldwcol,oldzerobinlen;
    int numrefcol, numnewcol, donewcolor,doemptyref;
    int thisseq,thisgroup,newgroup,numstart,numend,numregions;
    onemove trialmove,*thismove;
    GArray *moves;
    window *currwin;
    double totalweight,moverandom,currweight,oldprevweight,truefullbinweight,maxweight,newcolweight;
    GPtrArray *binwin1;

    
    if ((v->bin)->len==1)  /* no windows selected, return*/
        return 1;
   
    oldzerobinlen=((GPtrArray *)g_ptr_array_index((v->bin),0))->len;

    if (v->usedir!=0) 
        v->usedir=1;

    zer=0;

    oldprevweight=*logprevweight;

    /* count total number of windows in all colours */
    nwins=0;
    for (nb=1; nb<(v->bin)->len; nb++){
        binwin1=g_ptr_array_index((v->bin),nb);
        nwins=nwins+binwin1->len;
    }
   
    thiswinnum=gsl_rng_uniform_int(v->gslrand,nwins);

    nb=0;
    nwins=0;
    do {
        nb++;
        binwin1=g_ptr_array_index((v->bin),nb);
        nwins=nwins+binwin1->len;
    } while (nwins <= thiswinnum);
    nwins=nwins-binwin1->len;

    currwin=g_ptr_array_index(binwin1,thiswinnum-nwins);

    oldwcol=currwin->colour;
    assert(oldwcol>0);
    assert(!currwin->blocked);

    /***determine group that the window is from****/
    thisseq = g_array_index((currwin->seq),int,0);
    /***group of this sequence***/
    thisgroup = g_array_index((v->seqgroups),int,thisseq);

    /***start and end window of this group****/
    numstart = g_array_index((v->groupstarts),int,thisgroup);
    numend = g_array_index((v->groupstarts),int,thisgroup+1);
  
    /* background this window*/
    blankwindow(v,currwin);
    /**count number of bases in each color ****/
    makebinbasecount(v);

    /* calculate weight prior to move, but "corrected" by chemical * potential for window to be placed */
    *logprevweight=fullbinweight(v)-(v->beta)*(v->mu);
    

    /***count number of ref colors and new colors with at least 1 window***/
    numrefcol = 0;
    numnewcol = 0;
    newcolweight = 0.0;
    /**count number of colors with at least 1 window****/
    for(n=1;n<((v->bin)->len);++n)
      {
        nbin=bincount(v->bin,n);
        if(nbin>0)
          {
            if(n<=((v->priorbinbase)->len))
              ++numrefcol;
            else
              ++numnewcol;
          }
      }

    /**if there are priors****/
    if((v->priorbinbase)->len > 0)
      {
	/**color moves or not at maximum number of colors***/
	if(v->Ncolm > 0 || (numnewcol+numrefcol) < v->descol)
	  {
	    /**allowed to put window in empty ref col***
	    doemptyref =1;
	    /**if newcolors left we are alloed to put them****/
	    if(v->numtfs - numnewcol > 0)
	      {
		donewcolor = 1;
		newcolweight = log((double) (v->numtfs-numnewcol))/(v->beta);
	      }
	    /**allowed to fill empty ref but no more room for new color***/
	    else
	      {
		donewcolor = 0;
		newcolweight = 0;
	      }
	  }
	/**no new color and no filling empty ref***/
	else
	  {
	    doemptyref = 0;
	    donewcolor = 0;
	    newcolweight = 0;
	  }
      }
    /**No priors***/
    else
      {
	/**no newcolweigh***/
	newcolweight = 0;
	/**allow increase in color number when doing color moves or when not at maximum***/
	if(v->Ncolm > 0 || (numnewcol+numrefcol) < v->descol)
	  {
	    donewcolor = 1;
	    doemptyref  = 1;
	  }
	else
	  {
	    donewcolor = 0;
	    doemptyref = 0;
	  }
      }
    

    /* now sample ways of replacing the window from background, not
     * necessarily same colour, but could be same colour, same window again */
    moves=g_array_new(FALSE,FALSE,sizeof(onemove));

    maxweight = -1000000.0;
    /**run over all windows in group***/
    for(n=numstart;n<numend;++n)
      {
	currwin=&g_array_index(v->win,window,n);
	/* only uncoloured and unblocked windows */
	if ((currwin->colour == 0) && (!currwin->blocked) && (!currwin->permblocked)) 
	  {
	    for (ndir=0; ndir<=v->usedir; ndir++) 
	      {
		for (nc=1; nc<(v->bin)->len; nc++) 
		  {
		    /**put newcolour if colour free and this refbin is empty***/
		    if(doemptyref || (bincount(v->bin,nc) > 0))
		      {
			trialmove.win=currwin;
			trialmove.dir=ndir;
			currwin->dir=ndir;
			trialmove.oldcolour=zer;
			trialmove.newcolour=nc;
			
			trialmove.moveweight=moveweight(v,&trialmove);
			/**if putting first window in refcolor we have new color***/
			if(bincount(v->bin,nc) == 0)
			  trialmove.moveweight -= v->lambda;
			if (trialmove.moveweight>maxweight)
			  maxweight = trialmove.moveweight;
			g_array_append_val(moves,trialmove);
		      }
		  }
		/**if we want a new colour ****/
		if(donewcolor)
		  {
		    trialmove.win=currwin;
		    trialmove.dir=ndir;
		    currwin->dir=ndir;
		    trialmove.oldcolour=zer;
		    /**new color ***/
		    trialmove.newcolour=(v->bin)->len;
		    nc = (v->bin)->len;
		    /* store log of moveweight */
		    trialmove.moveweight= moveweight(v, &trialmove)-(v->lambda)+newcolweight;
		    if (trialmove.moveweight>maxweight)
		      maxweight = trialmove.moveweight;
		    g_array_append_val(moves,trialmove);
		  }
	      }
	  }
      }

    /***if we want to do an extra region, pick randomly another region*****/
    if(extraregion)
      {
	numregions = (v->groupstarts)->len-1;
	/**pick random number between 0 and numregions-2 starting count one after current region, mod numregions***/
	newgroup = (thisgroup+1+gsl_rng_uniform_int(v->gslrand,numregions-1)) % numregions;
	numstart = g_array_index((v->groupstarts),int,newgroup);
	numend = g_array_index((v->groupstarts),int,newgroup+1);
	/*printf("%d groups thisgroup %d newgroup %d\n",numregions,thisgroup,newgroup);*/
	if(newgroup == thisgroup)
	  printf("error: other group is equal to thisgroup\n");
	/***now do same procedure for all windows in this group****/
	for(n=numstart;n<numend;++n)
	  {
	    currwin=&g_array_index(v->win,window,n);
	    /* only uncoloured and unblocked windows */
	    if ((currwin->colour == 0) && (!currwin->blocked) && (!currwin->permblocked)) 
	      {
		for (ndir=0; ndir<=v->usedir; ndir++) 
		  {
		    for (nc=1; nc<(v->bin)->len; nc++) 
		      {
			/**put newcolour if colour free and this refbin is empty***/
			if(doemptyref || (bincount(v->bin,nc) > 0))
			  {
			    trialmove.win=currwin;
			    trialmove.dir=ndir;
			    currwin->dir=ndir;
			    trialmove.oldcolour=zer;
			    trialmove.newcolour=nc;
			    
			    trialmove.moveweight=moveweight(v,&trialmove);
			    /**if putting first window in refcolor we have new color***/
			    if(bincount(v->bin,nc) == 0)
			      trialmove.moveweight -= v->lambda;
			    if (trialmove.moveweight>maxweight)
			      maxweight = trialmove.moveweight;
			    g_array_append_val(moves,trialmove);
			  }
		      }
		    /**if we want a new colour ****/
		    if(donewcolor)
		      {
			trialmove.win=currwin;
			trialmove.dir=ndir;
			currwin->dir=ndir;
			trialmove.oldcolour=zer;
			/**new color ***/
			trialmove.newcolour=(v->bin)->len;
			nc = (v->bin)->len;
			trialmove.moveweight= moveweight(v,&trialmove)-(v->lambda)+newcolweight;
			if (trialmove.moveweight>maxweight)
			  maxweight = trialmove.moveweight;
			g_array_append_val(moves,trialmove);
		      }
		  }
	      }
	  }
      }
	
    /***calculate sum of the weights****/
    totalweight = 0.0;
    n = (moves->len);
    for (nm=0;nm<n;++nm) {
      thismove = &g_array_index(moves,onemove,nm);
      thismove->moveweight = exp((v->beta)*(thismove->moveweight-maxweight));
      totalweight += thismove->moveweight;
    }
    moverandom=gsl_rng_uniform(v->gslrand)*totalweight;
    
    /***sample a move****/
    currweight=0.0;
    nm=0;
    do {
        trialmove=g_array_index(moves,onemove,nm);
        currweight += trialmove.moveweight;
        nm++;
    } while (currweight<moverandom && nm < n);

    *logprevweight=*logprevweight + (v->beta)*maxweight + log(trialmove.moveweight);
    if(trialmove.newcolour == (v->bin)->len)
      {
        *logprevweight -= (v->beta)*newcolweight;
      }

    /* and update binset and windowset accordingly */
    colourwindow(v,&trialmove);
    
    g_array_free(moves,TRUE);

    /* We could update binbase cleanly */
    /* but don't bother -- this looks fast enough    */
    makebinbasecount(v);

    /* check that prevweight is being kept track of properly */

    truefullbinweight=fullbinweight(v);
    if(fabs(((*logprevweight)-truefullbinweight)) > 0.1)
      fprintf(stderr,"WARNING: abnormally large difference between the calculated score %lf\nand the true full score %lf in switchwindow\n",*logprevweight,truefullbinweight);
    *logprevweight = truefullbinweight;

    assert(oldzerobinlen==((GPtrArray *)g_ptr_array_index((v->bin),0))->len);

    return 0;
}    
