#include "procse.h"
#include "procse_State.h"
#include "procse_TempClock.h"
#include "procse_Record.h"
#include "procse_MonteCarlo.h"

/* LC (Log Cache) */
void LC::configure(int cachesize)
{
	if (cachesize > size) {
		delete[] cache;
		cache = new double[cachesize];
		if (!cache) {
			fprintf(stderr,"Error: out of memory in LC::configure\n");
			fflush(0);
			exit(1);
		}
		size = cachesize;
	}
	cache[0] = 0.0; /* defining log(0) to be 0.0 (as algoritmic shortcut) */
	for (int x = 1; x < cachesize; x++) cache[x] = ::log(x);
}

void LC::destroy()
{
	delete[] cache;
}

int LC::size = 1;
double *LC::cache = new double;

/* LGNC (Log Gamma Normalized Cache) implementation */

void LGNC::configure(int cachesize, double q)
{
	if (q <= 0.0 || q > 1.0) {
		fprintf(stderr,"Error: illegal q (%20f) used with Log Gamma Normalized Cache\n",q);
		exit(1);
	}
	if (cachesize > size) {
		delete[] cache_q;
		delete[] cache_4q;
		cache_q = new double[cachesize];
		cache_4q = new double[cachesize];
		if (!cache_q || !cache_4q) {
			fprintf(stderr,"Error: out of memory in LGNC::configure\n");
			fflush(0);
			exit(1);
		}
		size = cachesize;
	}
	cache_q[0] = 0.0;
	cache_4q[0] = 0.0;
	for (int x = 1; x < cachesize; x++) {
		cache_q[x] = log(x + q - 1.0) + cache_q[x - 1];
		cache_4q[x] = log(x + 4.0 * q - 1.0) + cache_4q[x - 1];
	}
}

void LGNC::destroy()
{
	delete[] cache_q;
	delete[] cache_4q;
}

int LGNC::size = 1;
double *LGNC::cache_q = new double;
double *LGNC::cache_4q = new double;

const char * const MainParameters::pname[] = {
	"random_seed",
	"q",
	"aw_len",
	"initial_cluster_size",
	"record_type",
	"entropy_method",
	"sizeadj",
	"chemical_potential",
	"coherent_shift_period",
	"transient_beta",
	"final_beta",
	"total_time_steps",
	"transient_time_steps",
	"deep_quench_time_steps",
	"significance_run_total_time_steps",
	"filename_prefix",
	"log_dump_period",
	"significance_sample_period",
	"sig_min_pair_presence",
	"cluster_member_sig_cutoff",
	"pseudocluster_cutoff",
	"verbose",
	"single_strand"
};

MainParameters::MainParameters()
{
	random_seed = time(0);
	q = 1;
	aw_len = -1;
	initial_cluster_size = 1;
	record_type = RT_SIGNIFICANCE;
	entropy_method = EM_PLAIN;
	sizeadj = SA_NONE;
	chemical_potential = 0.0;
	coherent_shift_period = 20.0;
	transient_beta = 1.0;
	final_beta = 4.0;
	total_time_steps = -1.0;
	transient_time_steps = 0.0;
	deep_quench_time_steps = 0.0;
	significance_run_total_time_steps = -1.0;
	filename_prefix[0] = '\0';
	log_dump_period = 3600;
	significance_sample_period = 1000.0;
	sig_min_pair_presence = 0.9;
	cluster_member_sig_cutoff = 0.05;
	pseudocluster_cutoff = 0.5;
	verbose = false;
	single_strand = false;
}

bool main_whitespace(int c)
{
	switch (c) {
	case ' ': case '\t': case '\v': case '\r': case '\n': case '\f': case '\a': case '\b':
		return true;
	}
	return false;
}

bool main_legalchar(int c)
{
	if (c >= 'a' && c <= 'z') return true;
	if (c >= 'A' && c <= 'Z') return true;
	if (c >= '0' && c <= '9') return true;
	if (c == '.' || c == '_' || c == '/') return true;
	return false;
}

int main_parse_int(const char *value)
{
	int i = 0;
	while (*value) {
		if (*value < '0' || *value > '9') {
			fprintf(stderr,"Error: while parsing parameter file, illegal character found\n");
			fprintf(stderr,"in integer value : %s\n",value);
			fflush(0);
			exit(1);
		}
		i *= 10;
		i += *value - '0';
		value++;
	}
	return i;
}

double main_parse_double(const char *value)
{
	double d = 0.0;
	while (*value) {
		if (*value < '0' || *value > '9') {
			if (*value == '.') {
				value++;
				break;
			}
			fprintf(stderr,"Error: while parsing parameter file, illegal character found\n");
			fprintf(stderr,"in integer value : %s\n",value);
			fflush(0);
			exit(1);
		}
		d *= 10.0;
		d += (double)(*value - '0');
		value++;
	}
	int decimal_places = 0;
	while (*value) {
		if (*value < '0' || *value > '9') {
			if (*value == '.') {
				fprintf(stderr,"Error: while parsing parameter file, more than one\n");
				fprintf(stderr,"decimal point in number: %s\n",value);
				fflush(0);
				exit(1);
			}
			fprintf(stderr,"Error: while parsing parameter file, illegal character found\n");
			fprintf(stderr,"in integer value : %s\n",value);
			fflush(0);
			exit(1);
		}
		decimal_places++;
		d *= 10.0;
		d += (double)(*value - '0');
		value++;
	}
	d = floor(d + 0.5); /* trim any rounding errors */
	d /= pow(10.0,(double)decimal_places);
	return d;
}

bool main_parse_bool(const char *value)
{
	if (strcmp(value,"true") == 0) return true;
	if (strcmp(value,"false") == 0) return false;
	fprintf(stderr,"Error: while parsing parameter file, expected 'true' or 'false' but found %s\n",value);
	fflush(0);
	exit(1);
}

void main_set_parameter(MainParameters *mp, const char *parameter, const char *value)
{
	static const int num_parameters = sizeof(mp->pname)/sizeof(mp->pname[0]);
	int p_id;
	for (p_id = 0; p_id < num_parameters; p_id++) {
		if (strcmp(parameter,mp->pname[p_id]) == 0) {
			/* match */
			switch (p_id) {
			case  0:
				mp->random_seed = main_parse_int(value);
				return;
			case  1:
				mp->q = main_parse_double(value);
				return;
			case  2:
				mp->aw_len = main_parse_int(value);
				return;
			case  3:
				mp->initial_cluster_size = main_parse_int(value);
				return;
			case  4:
				mp->record_type = (MainParameters::RecordType)main_parse_int(value);
				return;
			case  5:
				mp->entropy_method =
					(MainParameters::StateEntropyMethod)main_parse_int(value);
				return;
			case  6:
				mp->sizeadj = 
					(MainParameters::StateSizeAdjust)main_parse_int(value);
				return;
			case  7:
				mp->chemical_potential = main_parse_double(value);
				return;
			case  8:
				mp->coherent_shift_period = main_parse_double(value);
				return;
			case  9:
				mp->transient_beta = main_parse_double(value);
				return;
			case 10:
				mp->final_beta = main_parse_double(value);
				return;
			case 11:
				mp->total_time_steps = main_parse_double(value);
				return;
			case 12:
				mp->transient_time_steps = main_parse_double(value);
				return;
			case 13:
				mp->deep_quench_time_steps = main_parse_double(value);
				return;
			case 14:
				mp->significance_run_total_time_steps = main_parse_double(value);
				return;
			case 15:
				strcpy(mp->filename_prefix,value);
				return;
			case 16:
				mp->log_dump_period = main_parse_int(value);
				return;
			case 17:
				mp->significance_sample_period = main_parse_double(value);
				return;
			case 18:
				mp->sig_min_pair_presence = main_parse_double(value);
				return;
			case 19:
				mp->cluster_member_sig_cutoff = main_parse_double(value);
				return;
			case 20:
				mp->pseudocluster_cutoff = main_parse_double(value);
				return;
			case 21:
				mp->verbose = main_parse_bool(value);
				return;
			case 22:
			  mp->single_strand = main_parse_bool(value);
			  return;
			default:
				fprintf(stderr, "Error: internal logic error in main_set_parameter\n");
				fflush(0);
				exit(1);
			}
		}
	}
	fprintf(stderr,"Error: unrecognized parameter name in parameter file: '%s'\n",parameter);
	fprintf(stderr,"Allowable parameters are:\n");
	for (p_id = 0; p_id < num_parameters; p_id++) {
		fprintf(stderr,"%s\n",mp->pname[p_id]);
	}
	fflush(0);
	exit(1);
}

void main_parse_parameter_file_line(MainParameters *mp, FILE *mpfile)
{
	char parameter[mp->MAX_WORD_LEN];
	char value[mp->MAX_WORD_LEN];
	int wordpos;
	int charbuf;
	charbuf = fgetc(mpfile);
	if (charbuf == '#') {
		/* comment line -- skip all */
		while (charbuf != EOF && charbuf != '\n') charbuf = fgetc(mpfile);
		return;
	}
	/* skip leading whitespace */
	while (charbuf != EOF && charbuf != '\n' && main_whitespace(charbuf)) charbuf = fgetc(mpfile);
	if (charbuf == EOF || charbuf == '\n') return; /* no parameter name */
	/* get parameter name */
	wordpos = 0;
	while (charbuf != EOF && charbuf != '\n' && !main_whitespace(charbuf)) {
		if (!main_legalchar(charbuf)) {
			fprintf(stderr,"Error: illegal character (hex:%02X) in parameter file\n",charbuf);
			fflush(0);
			exit(1);
		}
		parameter[wordpos++] = (char)charbuf;
		if (wordpos >= mp->MAX_WORD_LEN) {
			parameter[wordpos] = '\0';
			fprintf(stderr,"Error: parameter name in parameter file is too long: %s\n",
					parameter);
			fflush(0);
			exit(1);
		}
		charbuf = fgetc(mpfile);
	}
	parameter[wordpos] = '\0';
	/* skip whitespace */
	while (charbuf != EOF && charbuf != '\n' && main_whitespace(charbuf))
		charbuf = fgetc(mpfile);
	if (charbuf == EOF || charbuf == '\n') {
		fprintf(stderr,"Error: no value given for parameter: %s\n",parameter);
		fflush(0);
		exit(1);
	}
	/* get parameter value */
	wordpos = 0;
	while (charbuf != EOF && charbuf != '\n' && !main_whitespace(charbuf)) {
		if (!main_legalchar(charbuf)) {
			fprintf(stderr,"Error: illegal character (hex:%02X) in parameter file\n",charbuf);
			fflush(0);
			exit(1);
		}
		value[wordpos++] = (char)charbuf;
		if (wordpos >= mp->MAX_WORD_LEN) {
			value[wordpos] = '\0';
			fprintf(stderr,"Error: parameter value in parameter file is too long: %s\n",
					value);
			fflush(0);
			exit(1);
		}
		charbuf = fgetc(mpfile);
	}
	value[wordpos] = '\0';
	/* skip anything else on this line */
	while (charbuf != EOF && charbuf != '\n') charbuf = fgetc(mpfile);
	/* set parameter value */
	main_set_parameter(mp, parameter, value);
}

void main_parse_parameter_file(MainParameters *mp, FILE *mpfile)
{
	while (!feof(mpfile)) main_parse_parameter_file_line(mp, mpfile);
	if (mp->significance_run_total_time_steps == -1.0) {
		mp->significance_run_total_time_steps = mp->total_time_steps;
	}
}

void main_check_parameters(MainParameters *mp, int trv_size)
{
	if (mp->q <= 0.0) {fprintf(stderr,"Error: q is non-positive\n");fflush(0);exit(1);}
	if (mp->aw_len < 1) {fprintf(stderr,"Error: aw_len is non-positive\n");fflush(0);exit(1);}
	if (mp->initial_cluster_size < 1) {
		fprintf(stderr,"Error: initial cluster size is non-positive\n");fflush(0);exit(1);
	}
	if (mp->initial_cluster_size > trv_size) {
		fprintf(stderr,"Error: initial cluster size bigger than number of TGA\n");fflush(0);exit(1);
	}
	if (mp->record_type != MainParameters::RT_NULL &&
		mp->record_type != MainParameters::RT_SIGNIFICANCE &&
		mp->record_type != MainParameters::RT_PAIRWISE) {
		fprintf(stderr,"Error: illegal value entered for record_type\n");fflush(0);exit(1);
	}
	if (mp->entropy_method != MainParameters::EM_PLAIN &&
		mp->entropy_method != MainParameters::EM_RANDOM) {
		fprintf(stderr,"Error: illegal value entered for entropy_method\n");fflush(0);exit(1);
	}
	if (mp->sizeadj != MainParameters::SA_NONE &&
		mp->sizeadj != MainParameters::SA_COUNT &&
		mp->sizeadj != MainParameters::SA_COUNT_AND_SIZE) {
		fprintf(stderr,"Error: illegal value entered for sizeadj\n");fflush(0);exit(1);
	}
	if (mp->coherent_shift_period <= 0.0) {
		fprintf(stderr,"Error: coherent shift period non-positive\n");fflush(0);exit(1);
	}
	if (mp->total_time_steps <= 0.0) {
		fprintf(stderr,"Error: total_time_steps is non-positive\n");fflush(0);exit(1);
	}
	if (mp->significance_run_total_time_steps <= 0.0) {
		fprintf(stderr,"Error: significance_run_total_time_steps is non-positive\n");fflush(0);exit(1);
	}
	if (mp->log_dump_period <= 0) {
		fprintf(stderr,"Error: log_dump_period is non-positive\n");fflush(0);exit(1);
	}
	if (mp->significance_sample_period <= 0.0) {
		fprintf(stderr,"Error: significance_sample_period is non-positive\n");fflush(0);exit(1);
	}
}

void main_print_parameters(FILE *out, MainParameters *mp)
{
	fprintf(out,"Parameters for run:\n");
	static const int num_parameters = sizeof(mp->pname)/sizeof(mp->pname[0]);
	int p_id;
	for (p_id = 0; p_id < num_parameters; p_id++) {
		fprintf(out, "%s ", mp->pname[p_id]);
		switch (p_id) {
		case  0:
			fprintf(out,"%d\n",mp->random_seed);
			break;
		case  1:
			fprintf(out,"%.20f\n",mp->q);
			break;
		case  2:
			fprintf(out,"%d\n",mp->aw_len);
			break;
		case  3:
			fprintf(out,"%d\n",mp->initial_cluster_size);
			break;
		case  4:
			fprintf(out,"%d\n",mp->record_type);
			break;
		case  5:
			fprintf(out,"%d\n",mp->entropy_method);
			break;
		case  6:
			fprintf(out,"%d\n",mp->sizeadj);
			break;
		case  7:
			fprintf(out,"%.20f\n",mp->chemical_potential);
			break;
		case  8:
			fprintf(out,"%.20f\n",mp->coherent_shift_period);
			break;
		case  9:
			fprintf(out,"%.20f\n",mp->transient_beta);
			break;
		case 10:
			fprintf(out,"%.20f\n",mp->final_beta);
			break;
		case 11:
			fprintf(out,"%.20f\n",mp->total_time_steps);
			break;
		case 12:
			fprintf(out,"%.20f\n",mp->transient_time_steps);
			break;
		case 13:
			fprintf(out,"%.20f\n",mp->deep_quench_time_steps);
			break;
		case 14:
			fprintf(out,"%.20f\n",mp->significance_run_total_time_steps);
			break;
		case 15:
			fprintf(out,"%s\n",mp->filename_prefix);
			break;
		case 16:
			fprintf(out,"%ld\n",mp->log_dump_period);
			break;
		case 17:
			fprintf(out,"%.20f\n",mp->significance_sample_period);
			break;
		case 18:
			fprintf(out,"%.20f\n",mp->sig_min_pair_presence);
			break;
		case 19:
			fprintf(out,"%.20f\n",mp->cluster_member_sig_cutoff);
			break;
		case 20:
			fprintf(out,"%.20f\n",mp->pseudocluster_cutoff);
			break;
		case 21:
			fprintf(out,"%s\n",mp->verbose?"true":"false");
			break;
		case 22:
		  fprintf(out,"%s\n",mp->single_strand?"true":"false");
		  break;
		default:
			fprintf(stderr, "Error: internal logic error in main_print_parameters\n");
			fflush(0);
			exit(1);
		}
	}
	fflush(0);
}

void main_init()
{
	srand(1);
}

MainParameters mp;

int main(int argc, char *argv[])
{
	/* check command line args */
	if (argc != 3) {
		printf("usage: %s <ASTAR input filename> <parameter filename>\n",argv[0]);
		exit(1);
	}
	/* read in TGA's from input file */
	FILE *in = fopen(argv[1],"r");
	static TGA_Raw_Vector trv(in);
	fclose(in);
	/* read in parameters from parameter file */
	in = fopen(argv[2],"r");
	if (!in) {
		fprintf(stderr, "Error: could not open paramenter file for reading\n");
		fflush(0);
		exit(1);
	}
	main_parse_parameter_file(&mp, in);
	fclose(in);
	main_check_parameters(&mp,trv.get_size());
	/* create log file */
	char log_filename[mp.MAX_FILENAME_PREFIX_LEN + 4];
	strcpy(log_filename,mp.filename_prefix);
	strcat(log_filename,"log");
	FILE *dump_log = fopen(log_filename,"w");
	if (!dump_log) {
		fprintf(stderr, "Error: could not open log file for writing\n");
		fflush(0);
		exit(1);
	}
	main_print_parameters(dump_log,&mp);
	/* initialize Log Gamma (Normalized) Cache */
	LGNC::configure(trv.get_total_tracts()+1,mp.q);
	TGA::set_aw_len(mp.aw_len);
	/* create start state */
	State *start_state;
	switch (mp.entropy_method) {
	case mp.EM_PLAIN:
		switch (mp.sizeadj) {
		case mp.SA_NONE:
			start_state = new StatePlainSANone(trv,mp.initial_cluster_size,mp.coherent_shift_period,mp.chemical_potential);
			break;
		case mp.SA_COUNT:
			start_state = new StatePlainSACount(trv,mp.initial_cluster_size,mp.coherent_shift_period,mp.chemical_potential);
			break;
		case mp.SA_COUNT_AND_SIZE:
			start_state = new StatePlainSACountAndSize(trv,mp.initial_cluster_size,mp.coherent_shift_period,mp.chemical_potential);
			break;
		default:
			fprintf(stderr, "Error: unknown sizeadj method in main()\n");
			fflush(0);
			exit(1);
		}
		break;
	case mp.EM_RANDOM:
		switch (mp.sizeadj) {
		case mp.SA_NONE:
			start_state = new StateRandomSANone(trv,mp.initial_cluster_size,mp.coherent_shift_period,mp.chemical_potential);
			break;
		case mp.SA_COUNT:
			start_state = new StateRandomSACount(trv,mp.initial_cluster_size,mp.coherent_shift_period,mp.chemical_potential);
			break;
		case mp.SA_COUNT_AND_SIZE:
			start_state = new StateRandomSACountAndSize(trv,mp.initial_cluster_size,mp.coherent_shift_period,mp.chemical_potential);
			break;
		default:
			fprintf(stderr, "Error: unknown sizeadj method in main()\n");
			fflush(0);
			exit(1);
		}
		break;
	default:
		fprintf(stderr, "Error: unknown entropy method in main()\n");
		fflush(0);
		exit(1);
	}
	if (!start_state) {
		fprintf(stderr, "Error: out of memory in main()\n");
		fflush(0);
		exit(1);
	}
	fprintf(dump_log,"start state:\n");
	start_state->write(dump_log,mp.verbose);
	State *ref_state = start_state->create_clone();
	/* create statistical record manager */
	Record *record;
	switch (mp.record_type) {
	case mp.RT_NULL:
		record = new RecordNull(dump_log,mp.log_dump_period);
		break;
	case mp.RT_SIGNIFICANCE:
		record = new RecordReference(dump_log,mp.log_dump_period,mp.q,ref_state,
				mp.cluster_member_sig_cutoff,mp.filename_prefix,
				mp.significance_sample_period,mp.sig_min_pair_presence);
		break;
	case mp.RT_PAIRWISE:
		record = new RecordPairwise(dump_log,mp.log_dump_period,mp.q,ref_state,
				mp.cluster_member_sig_cutoff,mp.filename_prefix,
				mp.pseudocluster_cutoff);
		break;
	default:
		fprintf(stderr, "Error: record type in main()\n");
		fflush(0);
		exit(1);
	}
	if (!record) {
		fprintf(stderr, "Error: out of memory in main()\n");
		fflush(0);
		exit(1);
	}
	/* begin processing */
	if (mp.record_type == mp.RT_PAIRWISE) {
		FixedTempClock tempclock(mp.transient_beta, mp.total_time_steps);
		monte_carlo_markov<State,MoveSet,FixedTempClock,Record>(ref_state,&tempclock,record);
		fprintf(dump_log,"final state:\n");
		ref_state->write(dump_log,mp.verbose);
	} else {
		ManualTempClock tempclock1(mp.transient_beta, mp.final_beta,
				mp.total_time_steps, mp.transient_time_steps, mp.deep_quench_time_steps);
		monte_carlo_markov<State,MoveSet,ManualTempClock,Record>(ref_state,&tempclock1,record);
		fprintf(dump_log,"reference state:\n");
		ref_state->write(dump_log,mp.verbose);
		if (mp.record_type == mp.RT_SIGNIFICANCE) {
			/* do best match significance sampling run */
			State *sig_state = start_state->create_clone();
			FixedTempClock tempclock2(1.0, mp.significance_run_total_time_steps);
			monte_carlo_markov<State,MoveSet,FixedTempClock,Record>(sig_state,&tempclock2,record);
		}
	}
	/* output statistics */
	record->write_output(ref_state,mp.verbose);
	delete record;
	delete start_state;
	fclose(dump_log);
	fflush(0);
	LC::destroy();
	LGNC::destroy();
	return 0;
}
