/* Source code courtesy of Jeffery D. Taft, PhD. */
/* http://www.nauticom.net/www/jdtaft/ */
/* modified and adapted 2003 */

#include "math.h"
#include "sym_eigen.h"

/**********************************************************************
  routine to tri-diagonalize a real symmetric matrix
                uses Householder's method
**********************************************************************/
void tri_diagonalize(double *Cxd, double *d,double *e, double *A, int L, double tol)
{
	int i, j, k, l;
	double f, g, h, hh;
	for (i = 0; i < L; i++) {
		for (j = 0; j <= i; j++) {
			A[i*L + j] = Cxd[i*L + j];
		}
	}
	for (i = L - 1; i > 0; i--) {
		l = i - 2;
		f = A[i*L + i - 1];
		g = 0.0;
		for (k = 0; k <= l; k++) {
			g += A[i*L + k]*A[i*L + k];
		}
		h = g + f*f;
		if (g <= tol) {
			e[i] = f;
			h = 0.0;
			d[i] = h;
			continue;
		}
		l++;
		g = sqrt(h);
		if (f >= 0.0) g = -g;
		e[i] = g;
		h = h - f*g;
		A[i*L + i - 1] = f - g;
		f = 0.0;
		for (j = 0; j <= l; j++) {
			A[j*L + i] = A[i*L + j]/h;
			g = 0.0;
			for (k = 0; k <= j; k++) {
				g += A[j*L + k]*A[i*L + k];
			}
			for (k = j + 1; k <= l; k++) {
				g += A[k*L + j]*A[i*L + k];				
			}
			e[j] = g/h;
			f += g*A[j*L + i];
		}
		hh = f/(h + h);
		for (j = 0; j <= l; j++) {
			f = A[i*L + j];
			g = e[j] - hh*f;
			e[j] = g;
			for (k = 0; k <= j; k++) {
				A[j*L + k] = A[j*L + k] - f*e[k] - g*A[i*L + k];
			}
		}
		d[i] = h;
	}
	d[0] = e[0] = 0.0;
	for (i = 0; i < L; i++) {
		l = i - 1;
		if (d[i] != 0.0) {
			for (j = 0; j <= l; j++) {
				g = 0.0;
				for (k = 0; k <= l; k++) {
					g += A[i*L + k]*A[k*L + j];
				}
				for (k = 0; k <= l; k++) {
					A[k*L + j] = A[k*L + j] - g*A[k*L + i];
				}
			}
		}
		d[i] = A[i*L + i];
		A[i*L + i] = 1.0;
		for (j = 0; j <= l; j++) {
			A[i*L + j] = A[j*L + i] = 0.0;
		}
	}
}

/**********************************************************************
    routine to find eigenstructure of real tri-diagonal matrix
			 uses QL algorithm
          returns  0: sucess      -1: failure to converge
**********************************************************************/
int calc_eigenstructure(double *d, double *e, double *A, int L, double macheps)
{
	int i, j, k, l, m;
	double b, c, f, g, h, p, r, s;

	for (i = 1; i < L; i++) e[i - 1] = e[i];
	e[L - 1] = b = f = 0.0;
	for (l = 0; l < L; l++) {
		h = macheps*(fabs(d[l]) + fabs(e[l]));
		if (b < h) b = h;
		for (m = l; m < L; m++) {
			if (fabs(e[m]) <= b) break;
		}
		j = 0;
		if (m != l) do {
			if (j++ == 30) return -1;
			p = (d[l + 1] - d[l])/(2.0*e[l]);
			r = sqrt(p*p + 1);
			h = d[l] - e[l]/(p + (p < 0.0 ? -r : r));
			for (i = l; i < L; i++) d[i] = d[i] - h;
			f += h;
			p = d[m];
			c = 1.0;
			s = 0.0;
			for (i = m - 1; i >= l; i--) {
				g = c*e[i];
				h = c*p;
				if (fabs(p) >= fabs(e[i])) {
					c = e[i]/p;
					r = sqrt(c*c + 1);
					e[i + 1] = s*p*r;
					s = c/r;
					c = 1.0/r;
				} else {
					c = p/e[i];
					r = sqrt(c*c + 1);
					e[i + 1] = s*e[i]*r;
					s = 1.0/r;
					c = c/r;
				}
				p = c*d[i] - s*g;
				d[i + 1] = h + s*(c*g + s*d[i]);
				for (k = 0; k < L; k++) {
					h = A[k*L + i + 1];
					A[k*L + i + 1] = s*A[k*L + i] + c*h;
					A[k*L + i] = c*A[k*L + i] - s*h;
				}
			}
			e[l] = s*p;
			d[l] = c*p;
		} while (fabs(e[l]) > b);
		d[l] = d[l] + f;
	}

/* order the eigenvectors  */
	for (i = 0; i < L; i++) {
		k = i;
		p = d[i];
		for (j = i + 1; j < L; j++) {
			if (d[j] < p) {
				k = j;
				p = d[j];
			}
		}
		if (k != i) {
			d[k] = d[i];
			d[i] = p;
			for (j = 0; j < L; j++) {
				p = A[j*L + i];
				A[j*L + i] = A[j*L + k];
				A[j*L + k] = p;
			}
		}
	}
	return 0;
}

#ifdef SYM_EIGEN_DRIVER
int main() {
	double startmatrix[3][3] = {	{0.0, 1.0, 2.0},
					{1.0, 0.0, 4.0},
					{2.0, 4.0, 0.0} };
	double d[3] = {0.0,0.0,0.0};
	double e[3];
	double triform[3][3];
	int i,j;
	tri_diagonalize(*startmatrix,d,e,*triform,3,1.0e-6);
	calc_eigenstructure(d,e,*triform,3,1.0e-18);
	printf("eigenvalues:\n");
	for (i=0; i<3; i++) printf("%2.4f\t",d[i]);
	printf("\n");
	printf("eigenvectors:\n");
	for (i=0; i<3; i++) {
		for (j=0; j<3; j++) {
			printf("%2.4f\t",triform[i][j]);
		}
		printf("\n");
	}
	return 0;
}
#endif
