class Record
{
protected:
	FILE * const log_file;
private:
	const long int log_dump_period;
	time_t last_log_time;
protected:
	inline void log_dump_check(State *state, double timestep);
public:
	Record(FILE *a_log_file, long int a_log_dump_period);
	virtual ~Record() {}
	virtual void update(State *state, double timestep) = 0;
	virtual void finalize_reference() = 0;
	virtual void write_output(State *state, bool verbose) = 0;
};

inline void Record::log_dump_check(State *state, double timestep)
{
        if (!(state->consistant())) {
                fprintf(stderr,"Error : statistics record updated in inconsistant state\n");
                exit(1);
        }
        time_t now = time(0);
#define DONT_TRACK_CLUSTER_SIZES
#ifdef TRACK_CLUSTER_SIZES
        if (difftime(now,last_log_time) >= 30) {
                fprintf(log_file, "Periodic Cluster Size dump at timestep %20f:\n\n",timestep);
                state->dump_cluster_sizes(log_file);
#else
        if (difftime(now,last_log_time) >= log_dump_period) {
                fprintf(log_file, "Periodic MCM State update at timestep %20f:\n\n",timestep);
                state->write(log_file,false);
#endif
                fflush(0);
                last_log_time = now;
        }
}

class RecordNull : public Record
{
	RecordNull(); /* no default constructor */
	RecordNull(const RecordNull &); /* no copy constructor */
	RecordNull &operator=(const RecordNull &); /* no assignment */
public:
	RecordNull(FILE *a_log_file, long int a_log_dump_period) :
		Record(a_log_file, a_log_dump_period) {}
	virtual ~RecordNull() {}
	virtual void update(State *state, double timestep) {log_dump_check(state, timestep);}
	virtual void finalize_reference() {}
	virtual void write_output(State *state, bool verbose) {}
};

class RecordWithWeightMatrix : public Record
{
protected:
	const double q;
	const int num_tga;			/* in state */
	const double cluster_member_sig_cutoff; /* stop listing members if less */
	const char *filename_prefix;		/* for opening cluster files */
	int *sorted_tga_id;			/* 2-d array of sorted tga id's per cluster */
	double *sorted_tga_sig;			/* 2-d array of sorted tga signif. per cl. */
	double *weight_matrix;			/* 3-d array of weight matricies per cl. */
	int *tga_aw_left;			/* 2-d array of tga aw_left per cl. */
	bool *tga_aw_sense;			/* 2-d array of tga aw_sense per cl. */
	double *final_entropy;			/* cluster entropies after reconstruction */
	double *final_information_score;	/* Information score per cluster */
	int *final_information_rank;		/* Used to pick final cluster numbering */
	void calculate_information_scores();
	void calculate_information_ranks();
	void output_information_ranks();
public:
	RecordWithWeightMatrix(FILE *a_log_file, long int a_log_dump_period,
		double a_q, int a_num_tga, double a_cluster_member_sig_cutoff,
		const char *a_filename_prefix);
	~RecordWithWeightMatrix();
};

class TGAClusterNode
{
public:
	int tga_id;
	double join_time;
	TGAClusterNode *previous;
	TGAClusterNode *next;
	TGAClusterNode(int a_tga_id);
};

class RecordReference : public RecordWithWeightMatrix
{
	static const int PURITY_HIST_BINS;
	State *last_sample_state;
	State *ref_state;
	double next_sample_time;	/* next timestep to sample on */
	double num_samples;
	int num_ref_clusters;
	double **cluster_sig_presence_hist;
	double **cluster_sig_purity_hist;
	double *cluster_sig_tga;
	const double significance_sample_period;
	const double sig_min_pair_presence;	/* cluster not detailed if less */
	RecordReference(); /* no default constructor */
	RecordReference(const RecordReference &); /* no copy constructor */
	RecordReference &operator=(const RecordReference &); /* no assignment */
public:
	RecordReference(FILE *a_log_file, long int a_log_dump_period,
		double a_q, State *state, double a_cluster_member_sig_cutoff,
		const char *a_filename_prefix, double a_significance_sample_period,
		double a_sig_min_pair_presence);
	virtual ~RecordReference();
	virtual void update(State *state, double timestep);
	virtual void finalize_reference();
	virtual void write_output(State *state, bool verbose);
	void output_cluster_significance_statistics();
	void calculate_tga_significance_statistics();
	void calculate_cluster_weight_matricies();
	void output_tga_significance_statistics();
	void output_cluster_weight_matricies(bool verbose);
};

class RecordPairwise : public RecordWithWeightMatrix
{
	double last_timestep;		/* last timestep seen by update() */
	TGAClusterNode **tga_list;
	TGAClusterNode **cluster_list;
	double *pair_duration;
	const double pseudocluster_cutoff;
	int *tga_pseudocluster;
	int *pseudocluster_size;
	double *anchor_prob;		/* 2-d array for holding ancor probs per psdoclstr */
	double *assoc_prob;		/* 2-d array for holding assoc probs per psdoclstr */
	void recursive_assign(int, int, double);	/* used for cutoff table */
	void output_cutoff_cluster_count(FILE *,double);/* used for cutoff table */
	void assign_to_pseudoclusters(double);	/* used for cutoff table */
	void output_pairwise_statistics();
	void output_pairwise_cutoff_table();
	void output_pseudocluster(FILE *out, State *state, double cutoff);
	void output_pseudoclusters(FILE *out, State *state);
	void calculate_pseudocluster_statistics(State *state);
	void calculate_pseudocluster_weight_matricies(State *state);
	void output_pseudocluster_statistics(State *state, bool verbose);
	void output_pseudocluster_weight_matricies(State *state, bool verbose);
	void arrange_reference_from_pairwise_data(State *state, double cutoff);
	RecordPairwise(); /* no default constructor */
	RecordPairwise(const RecordPairwise &); /* no copy constructor */
	RecordPairwise &operator=(const RecordPairwise &); /* no assignment */
public:
	RecordPairwise(FILE *a_log_file, long int a_log_dump_period,
		double a_q, State *state, double a_cluster_member_sig_cutoff,
		const char *a_filename_prefix, double a_pseudocluster_cutoff);
	virtual ~RecordPairwise();
	virtual void update(State *state, double timestep);
	virtual void finalize_reference();
	virtual void write_output(State *state, bool verbose);
};

