class TGA_Raw;				/* tract group data from file */
class TGA_Raw_Vector;			/* array of TGA_Raw */
class TGA;				/* reference to TGA_Raw plus alignment window */
class Cluster;				/* group of TGA */
class MoveSet;				/* abstraction of a set of moves */
class InertMoveSet;			/* a set of possible inserts to be chosen from */
class CoherentShiftMoveSet;		/* a set of possible cluster wide shifts of alignment */
class State;				/* group of Cluster */

class TGA_Raw
{
public:
	enum {NUM_BASES = 4};		/* A, C, G, T */
private:
	char *name;			/* from user input file */
	int id;				/* TGA id number */
	int num_tracts;
	int tract_len;			/* length of the member tracts */
	int (*base_distrib)[NUM_BASES];	/* 2D array, sums [position][base] over member tracts */
	TGA_Raw();			/* no default constructor */
	TGA_Raw(const TGA_Raw &);	/* no copy constructor */
	TGA_Raw &operator=(const TGA_Raw &);	/* no assign */
public:
	TGA_Raw(const char a_name[], int a_id, int a_num_tracts, int tract_len,
			int a_base_distrib[][NUM_BASES]);
	~TGA_Raw();
	const char *get_name() const {return name;}
	int get_id() const {return id;}
	int get_num_tracts() const {return num_tracts;}
	int get_tract_len() const {return tract_len;}
	const int (*(get_distrib() const))[NUM_BASES] {return base_distrib;}
	const int *distrib_slice(int pos) const {return base_distrib[pos];}
};

class TGA_Raw_Vector
{
	TGA_Raw **tga;			/* for array access to items */
	int size;			/* number of TGA elements */
	TGA_Raw_Vector();		/* no default constructor */
	TGA_Raw_Vector(const TGA_Raw_Vector &);	/* no copy constructor */
	TGA_Raw_Vector &operator=(const TGA_Raw_Vector &); /* no assign */
public:
	TGA_Raw_Vector(FILE *in);
	~TGA_Raw_Vector();
	const TGA_Raw *operator[](int index) const;
	int get_size() const {return size;}
	int get_total_tracts() const;
};

class TGA
{
	const TGA_Raw *core;
	static int aw_len;		/* alignment window length */
	static int aw_len_max;		/* maximum alignment window length (length of smallest tract) */
	int aw_left_max;		/* maximum allowable aw_left */
	TGA();				/* no default constructor */
	TGA(const TGA &);		/* no copy constructor */
	TGA(const TGA *tga);		/* deep copy constructor */
public:
	int aw_left;			/* left edge offset of the alignment window */
	bool aw_sense;			/* true = read left to right, false = right to left */
	int cluster_num;		/* index number of cluster */
	TGA(const TGA_Raw *a_core);
	TGA *create_clone();		/* returns dynamic memory clone */
	static void tract_len_seen(int len);
	static void set_aw_len(int new_len);
	static int get_aw_len() {return aw_len;}
	const char *get_name() const {return core->get_name();}
	int get_id() const {return core->get_id();}
	int get_num_tracts() const {return core->get_num_tracts();}
	int get_tract_len() const {return core->get_tract_len();}
	const int (*(get_distrib() const))[TGA_Raw::NUM_BASES] {return core->get_distrib();}
	const int (*(get_aw_distrib() const))[TGA_Raw::NUM_BASES] {return core->get_distrib() + aw_left;}
	int get_aw_left_max() const {return aw_left_max;}
	void write(FILE *out, bool verbose, int aw_left_max_max); /* pass global aw_left_max to align columns */
};

class Cluster
{
	Cluster();			/* no default constructor */
	Cluster(const Cluster &);	/* no copy constructor */
	Cluster &operator=(const Cluster &);	/* no assign */
protected:
	int id;				/* cluster id number */
	int num_tga;			/* number of tga members */
	int num_tracts;			/* total number of tracts in member tga's */
	int (*base_distrib)[TGA_Raw::NUM_BASES];	/* array to hold base distrib of alignment */
	double entropy;			/* cluster entropy */
	Cluster(const Cluster *cluster);	/* deep copy constructor */
public:
	Cluster(int a_id);		/* gives empty cluster */
	virtual Cluster *create_clone() = 0; /* returns dynamic memory clone */
	virtual ~Cluster();
	bool empty() const {return num_tga == 0;}
	bool singlet() const {return num_tga == 1;}
	int get_id() const {return id;}
	int get_num_tga() const {return num_tga;}
	int get_num_tracts() const {return num_tracts;}
	virtual void get_insert_entropy_profile(TGA *tga, double *alignment_entropy, int *vector_pos) const = 0;
	virtual void insert(TGA *tga) = 0;		/* insert distribution and calculate entropy */
	void insert(TGA *tga, double new_entropy);	/* insert distr. and set entropy */
	virtual void remove(TGA *tga) = 0;
	void set_distrib_and_entropy(int *distrib_start, double a_entropy); /* replaces contents */
	double get_entropy() const {return entropy;}
	void write(FILE *out, int aw_left_max_max); /* pass global aw_left_max to align columns */
};

class ClusterPlain : public Cluster
{
	ClusterPlain(const ClusterPlain *cluster);	/* deep copy constructor */
public:
	ClusterPlain(int a_id);
	virtual ~ClusterPlain() {}
	virtual Cluster *create_clone();
	virtual void get_insert_entropy_profile(TGA *tga, double *alignment_entropy, int *vector_pos) const;
	virtual void insert(TGA *tga);		/* insert distribution and calculate entropy */
	virtual void remove(TGA *tga);
};

class ClusterRandom : public Cluster
{
	ClusterRandom(const ClusterRandom *cluster);	/* deep copy constructor */
public:
	ClusterRandom(int a_id);
	virtual ~ClusterRandom() {}
	virtual Cluster *create_clone();
	virtual void get_insert_entropy_profile(TGA *tga, double *alignment_entropy, int *vector_pos) const;
	virtual void insert(TGA *tga);		/* insert distribution and calculate entropy */
	virtual void remove(TGA *tga);
};

class MoveSet
{
	MoveSet();			/* no default constructor */
	MoveSet(const MoveSet &);       /* no copy constructor */
	MoveSet &operator=(const MoveSet &); /* no assign */
protected:
	const int state_size;		/* size from state */
	int num_moves;			/* total number of alignments */
	double *neglogpostprob;		/* negative log probability for each possible move */
	int neglogpostprob_size;
	bool applied;
public:
	MoveSet(int a_state_size);
	virtual ~MoveSet();
	int get_num_moves() const {return num_moves;}
	virtual void apply(int take_move) = 0;
	bool completed() const {return applied;}
	double *get_neglogpostprob() const {return neglogpostprob;}
};

class InsertMoveSet : public MoveSet
{
	InsertMoveSet();		/* no default constructor */
	InsertMoveSet(const InsertMoveSet &);	/* no copy constructor */
	InsertMoveSet &operator=(const InsertMoveSet &); /* no assign */
protected:
	TGA *tga;			/* the tga which is being transferred */
	Cluster *donor;			/* cluster which started with TGA */
	ReVector<Cluster *> acceptor;	/* vector of possible destinations for TGA */
	double *inserted_entropy;	/* array of entropies for all possible alignments */
	int inserted_entropy_size;
	Cluster *destination;		/* cluster where TGA ended */
	int *cluster_size_hist;		/* cluster size histogram */
	const double * const stirling_partition;	/* used for Count SizeAdj */
	const double chemical_potential;
	virtual double get_prior_nlpp(Cluster *cluster) const = 0;
public:
	InsertMoveSet(int a_state_size, int *a_cluster_size_hist,
		double *a_stirling_partition, double a_chemical_potential);
	~InsertMoveSet();
	void canabalize(Cluster *a_donor, Cluster *a_acceptor, TGA *a_tga);
	virtual void apply(int take_move);	/* modify state according to move */
	bool move_made() const {return destination != donor;}
	int moved_tga() const {return tga->get_id();}
	int moved_tga_donor() const {return donor->get_id();}
	int moved_tga_acceptor() const {return destination->get_id();}
};

class InsertMoveSetSANone : public InsertMoveSet
{
	virtual double get_prior_nlpp(Cluster *cluster) const;
public:
	InsertMoveSetSANone(int a_state_size, int *a_cluster_size_hist,
		double *a_stirling_partition, double a_chemical_potential);
};

class InsertMoveSetSACount : public InsertMoveSet
{
	virtual double get_prior_nlpp(Cluster *cluster) const;
public:
	InsertMoveSetSACount(int a_state_size, int *a_cluster_size_hist,
		double *a_stirling_partition, double a_chemical_potential);
};

class InsertMoveSetSACountAndSize : public InsertMoveSet
{
	virtual double get_prior_nlpp(Cluster *cluster) const;
public:
	InsertMoveSetSACountAndSize(int a_state_size, int *a_cluster_size_hist,
		double *a_stirling_partition, double a_chemical_potential);
};

class CoherentShiftMoveSet : public MoveSet
{
	CoherentShiftMoveSet();		/* no default constructor */
	CoherentShiftMoveSet(const CoherentShiftMoveSet &);	/* no copy constructor */
	CoherentShiftMoveSet &operator=(const CoherentShiftMoveSet &);	/* no assign */
protected:
	TGA **&tga;			/* reference to state's tgas */
	Cluster *cluster;		/* cluster to shift */
	int num_member_tga;		/* number of tga in cluster */
	ReVector<TGA *> member_tga;	/* tga members of cluster */
	int max_left_shift;		/* furthest shift of windows to the left */
	int base_distrib_width;		/* number of columns in the expanded distribution */
	ReVector<int[TGA_Raw::NUM_BASES]> base_distrib; /* expanded distribution of cluster */
	double *column_entropy;		/* entropy of a column in the alignment */
	int column_entropy_size;
	void canabalize_fill_base_distrib(); /* first half of canabalize(Cluster *) */
	void canabalize_set_neglogpostprob(); /* second half of canabalize(Cluster *) */
public:
	CoherentShiftMoveSet(int a_state_size, TGA **&a_tga);
	virtual ~CoherentShiftMoveSet();
	virtual void canabalize(Cluster *a_cluster) = 0;
	virtual void apply(int take_move);	/* modify state according to move */
};

class CoherentShiftMoveSetPlain : public CoherentShiftMoveSet
{
	CoherentShiftMoveSetPlain();	/* no default constructor */
	CoherentShiftMoveSetPlain (const CoherentShiftMoveSetPlain &);	/* no copy constructor */
	CoherentShiftMoveSetPlain &operator=(const CoherentShiftMoveSetPlain &); /* no assign */
public:
	CoherentShiftMoveSetPlain(int a_state_size, TGA **&a_tga);
	virtual ~CoherentShiftMoveSetPlain() {}
	virtual void canabalize(Cluster *a_cluster);
};

class CoherentShiftMoveSetRandom : public CoherentShiftMoveSet
{
	CoherentShiftMoveSetRandom();	/* no default constructor */
	CoherentShiftMoveSetRandom (const CoherentShiftMoveSetRandom &);	/* no copy constructor */
	CoherentShiftMoveSetRandom &operator=(const CoherentShiftMoveSetRandom &); /* no assign */
public:
	CoherentShiftMoveSetRandom(int a_state_size, TGA **&a_tga);
	virtual ~CoherentShiftMoveSetRandom() {}
	virtual void canabalize(Cluster *a_cluster);
};

class State
{
	State();			/* no default constructor */
	State(const State&);		/* no copy constructor */
	State &operator=(const State&);	/* no assign */
protected:
	int size;			/* number of tga's *and* clusters */
	TGA **tga;			/* for array access to tga's */
	Cluster **cluster;		/* for array access to clusters */
	bool insertmove;		/* true if last move was an insertmove */
	InsertMoveSet *insertmoveset;	/* to hold reusable move object */
	CoherentShiftMoveSet *coherentshiftmoveset; /* to hold reusable move object */
	const double coherentshiftperiod;
	double coherentshiftcount;	/* count down to Coherent Shift */
	int *cluster_size_hist;		/* cluster size histogram */
	double *stirling_partition;	/* used in Count size adjustment */
	const double chemical_potential;
	virtual double get_column_entropy(const double *column, double q) const = 0;
	double get_column_entropy_plain(const double *column, double q) const;
	double get_column_entropy_random(const double *column, double q) const;
	State(const State *state);	/* deep copy constructor */
public:
	State(const TGA_Raw_Vector &trv,
		int initial_cluster_size,
		double a_coherent_shift_period,
		double a_chemical_potential);
	virtual State *create_clone() = 0; /* returns dynamic memory clone */
	virtual ~State();
	int get_num_tga() const {return size;}
	int get_num_cluster() const;
	int get_cluster_size(int cluster_id) const {return (cluster_id >= size ? 0 : cluster[cluster_id]->get_num_tga());}
	double get_entropy() const;
	MoveSet *propose_moveset();
	bool consistant() const {return insertmoveset->completed();}
	bool move_made() const {return insertmove && insertmoveset->move_made();}
	int moved_tga() const {return insertmoveset->moved_tga();}
	int moved_tga_donor() const {return insertmoveset->moved_tga_donor();}
	int moved_tga_acceptor() const {return insertmoveset->moved_tga_acceptor();}
	const char *get_tga_name(int tga_id) const {return (tga_id >= size ? "Err" : tga[tga_id]->get_name());}
	int get_tga_cluster(int tga_id) const {return (tga[tga_id]->cluster_num);}
	void rearrange_to_pairwise_assignment(const int *assignments);
	void accumulate_significance_metrics(State *state,
		double **cluster_sig_presence_hist,
		double **cluster_sig_purity_hist,
		double *tga_sig);
	void write(FILE *out, bool verbose);
	void dump_cluster_sizes(FILE *out);
	void write_membership_summary(FILE *out);
	void calculate_cluster_weight_matrix(double *weight_matrix,
			int *tga_aw_left, bool *tga_aw_sense, double *final_entropy,
			const int *sorted_tga_id, const double *sorted_sig, double q) const;
	void write_cluster_weight_matrix(const double *weight_matrix,
			const int *tga_aw_left, const bool *tga_aw_sense, const double *final_entropy,
			FILE *out, bool verbose, const int *sorted_tga_id, const double *sorted_sig, double q) const;
};

class StatePlainSANone : public State
{
	StatePlainSANone(const StatePlainSANone *state);	/* deep copy constructor */
	virtual double get_column_entropy(const double *column,double q) const {return get_column_entropy_plain(column,q);}
public:
	StatePlainSANone(const TGA_Raw_Vector &trv,
		int initial_cluster_size,
		double a_coherent_shift_period,
		double a_chemical_potential);
	virtual ~StatePlainSANone();
	virtual State *create_clone();
};

class StatePlainSACount : public State
{
	StatePlainSACount(const StatePlainSACount *state);	/* deep copy constructor */
	virtual double get_column_entropy(const double *column,double q) const {return get_column_entropy_plain(column,q);}
public:
	StatePlainSACount(const TGA_Raw_Vector &trv,
		int initial_cluster_size,
		double a_coherent_shift_period,
		double a_chemical_potential);
	virtual ~StatePlainSACount();
	virtual State *create_clone();
};

class StatePlainSACountAndSize : public State
{
	StatePlainSACountAndSize(const StatePlainSACountAndSize *state);	/* deep copy constructor */
	virtual double get_column_entropy(const double *column,double q) const {return get_column_entropy_plain(column,q);}
public:
	StatePlainSACountAndSize(const TGA_Raw_Vector &trv,
		int initial_cluster_size,
		double a_coherent_shift_period,
		double a_chemical_potential);
	virtual ~StatePlainSACountAndSize();
	virtual State *create_clone();
};

class StateRandomSANone : public State
{
	StateRandomSANone(const StateRandomSANone *state);/* deep copy constructor */
	virtual double get_column_entropy(const double *column,double q) const {return get_column_entropy_random(column,q);}
public:
	StateRandomSANone(const TGA_Raw_Vector &trv,
		int initial_cluster_size,
		double a_coherent_shift_period,
		double a_chemical_potential);
	virtual ~StateRandomSANone();
	virtual State *create_clone();
};

class StateRandomSACount : public State
{
	StateRandomSACount(const StateRandomSACount *state);	/* deep copy constructor */
	virtual double get_column_entropy(const double *column,double q) const {return get_column_entropy_random(column,q);}
public:
	StateRandomSACount(const TGA_Raw_Vector &trv,
		int initial_cluster_size,
		double coherent_shift_period,
		double a_chemical_potential);
	virtual ~StateRandomSACount();
	virtual State *create_clone();
};

class StateRandomSACountAndSize : public State
{
	StateRandomSACountAndSize(const StateRandomSACountAndSize *state);	/* deep copy constructor */
	virtual double get_column_entropy(const double *column,double q) const {return get_column_entropy_random(column,q);}
public:
	StateRandomSACountAndSize(const TGA_Raw_Vector &trv,
		int initial_cluster_size,
		double coherent_shift_period,
		double a_chemical_potential);
	virtual ~StateRandomSACountAndSize();
	virtual State *create_clone();
};
