#define DEEP_QUENCH_BETA 25.0

class FixedTempClock
{
	enum Phase { TRANSIENT, DONE };
	Phase phase;
	double current_time;
	const double end_time;
	const double beta;
	FixedTempClock(); /* no default constructor */
	FixedTempClock(const FixedTempClock &); /* no copy constructor */
	FixedTempClock &operator=(const FixedTempClock &); /* no assign */
public:
	FixedTempClock(double a_beta, double total_time_steps) :
		phase(TRANSIENT), current_time(0.0),
		end_time(total_time_steps), beta(a_beta) {}
	double get_beta() const {return beta;}
	void step() {
		current_time += 1.0;
		if (current_time >= end_time) {
			phase = DONE;
			current_time -= 1.0;
		}
	}
	double get_time() const {return current_time;}
	bool done() const {return phase == DONE;}
};

class ManualTempClock
{
	enum Phase { TRANSIENT, ANNEAL, DEEP_QUENCH, DONE };
	Phase phase;
	double current_time;
	const double anneal_begin_time;
	const double deep_quench_begin_time;
	const double end_time;
	const double start_beta;
	double beta;
	const double end_beta;
	const double alpha;
	ManualTempClock(); /* no default constructor */
	ManualTempClock(const ManualTempClock &); /* no copy constructor */
	ManualTempClock &operator=(const ManualTempClock &); /* no assign */
public:
	double get_beta() const {return beta;}
	ManualTempClock(double transient_beta, double final_beta,
		double total_time_steps, double transient_time_steps,
		double deep_quench_time_steps) :
		phase(TRANSIENT), current_time(0.0), anneal_begin_time(transient_time_steps),
		deep_quench_begin_time(total_time_steps - deep_quench_time_steps),
		end_time(total_time_steps), start_beta(transient_beta), beta(transient_beta),
		end_beta(final_beta),
		alpha((final_beta - transient_beta) / (deep_quench_begin_time - anneal_begin_time)) {}
	void step()
	{
		current_time += 1.0;
		if (current_time < anneal_begin_time) return;
		if (current_time < deep_quench_begin_time) {
			phase = ANNEAL;
			beta = start_beta + alpha * (current_time - anneal_begin_time);
			return;
		}
		if (current_time < end_time) {
			phase = DEEP_QUENCH;
			beta = DEEP_QUENCH_BETA;
			return;
		}
		phase = DONE;
		current_time -= 1.0;
	}
	double get_time() const {return current_time;}
	bool done() const {return phase == DONE;}
};
	
#if 0
/* AutoTempClock is still not working correctly for large
data sets; The entropy log file can take overly long to
fill up -- ensuring a long transient period. Tuning and
thought is still needed to make this work.
*/
class AutoTempClock
{
	enum Phase { TRANSIENT, ANNEAL, DEEP_QUENCH, DONE };
	enum {ENTROPY_LOG_LEN = 20};
	enum {TRANSIENT_DOWNTICK_THRESHOLD = 7};
	Phase phase;
	long int current_time;
	const long int desired_seconds;
	const time_t clock_start;
	double entropy_log[ENTROPY_LOG_LEN];
	const int entropy_log_period;
	long int anneal_begin_time;
	long int deep_quench_begin_time;
	long int end_time;
	const double start_beta;
	double beta;
	const double end_beta;
	double alpha;
	AutoTempClock(); /* no default constructor */
	AutoTempClock(const AutoTempClock &); /* no copy constructor */
	AutoTempClock &operator=(const AutoTempClock &); /* no assign */
public:
	double get_beta() const {return beta;}
	AutoTempClock(State *state, long int seconds,
		double final_beta, double transient_beta) :
		phase(TRANSIENT), current_time(0L), desired_seconds(seconds), clock_start(time(0)),
		entropy_log_period(state->get_num_tga() * state->get_num_tga() / 1000),
		anneal_begin_time(LONG_MAX), deep_quench_begin_time(LONG_MAX), end_time(LONG_MAX),
		start_beta(transient_beta), beta(transient_beta), end_beta(final_beta), alpha(0) {
		int entropy_log_pos;
		for (entropy_log_pos = 0; entropy_log_pos < ENTROPY_LOG_LEN; entropy_log_pos++) {
			entropy_log[entropy_log_pos] = DBL_MAX/(entropy_log_pos+1); /* decreasing */
		}
	}
	void step(State *state) {
		current_time++;
		switch (phase) {
		case ANNEAL:
			if (current_time == deep_quench_begin_time) {
				phase = DEEP_QUENCH;
				beta = DEEP_QUENCH_BETA;
				end_time = current_time + 100 * state->get_num_tga() * state->get_num_cluster();
			} else {
				beta = start_beta + alpha * (current_time - anneal_begin_time);
			}
			break;
		case TRANSIENT:
			if (current_time % entropy_log_period == 0) {
				int e_log_pos = 0;
				int down_count = 0;
				for (; e_log_pos < ENTROPY_LOG_LEN - 1; e_log_pos++) {
					if (entropy_log[e_log_pos] > entropy_log[e_log_pos + 1]) down_count++;
					entropy_log[e_log_pos] = entropy_log[e_log_pos+1];
				}
				entropy_log[ENTROPY_LOG_LEN - 1] = state->get_entropy();
				if (entropy_log[ENTROPY_LOG_LEN - 2] >
						entropy_log[ENTROPY_LOG_LEN - 1]) down_count++;
				if (down_count > TRANSIENT_DOWNTICK_THRESHOLD) break;
				double clock_period = difftime(time(0),clock_start);
				if (clock_period < 1.0) break;
				phase = ANNEAL;
				anneal_begin_time = current_time;
				const long int steps_per_second = (long int)(current_time / clock_period);
				deep_quench_begin_time = steps_per_second * desired_seconds;
				alpha = (end_beta - start_beta) /
						(deep_quench_begin_time - anneal_begin_time);
				if (current_time >= deep_quench_begin_time) {
					fprintf(stderr,"WARNING: insufficiant time for any annealing to occur!\n");
					fprintf(stderr,"         deep quench will immediately follow transient period\n");
					phase = DEEP_QUENCH;
					beta = DEEP_QUENCH_BETA;
					end_time = current_time + 100 * state->get_num_tga() * state->get_num_cluster();
				}
			}
			break;
		case DEEP_QUENCH:
			if (current_time == end_time) {
				phase = DONE;
			}
			break;
		default: /* DONE */
			current_time--;
			break;
		}
	}
	long int get_time() const {return current_time;}
	bool done() const {return phase == DONE;}
};
#endif

