/* spa.h */

/* Locus - an indexed segement of a DNA strand */
struct Locus;

/* struct diagSeg - An extended segment of defined positions along a diagonal  */
struct diagSeg
{
	struct diagSeg *next;		/* link for list */
	unsigned int qStart, qEnd;	/* Positions in query. */
	unsigned int tStart, tEnd;	/* Positions in target. */
	unsigned int diagonal;
};

/* struct Cell - Each individual cell in an alignment */
struct Cell
{
	struct Cell *parent;
	struct Cell *child;
	struct Cell *near_neighbour;
	unsigned int clone_pos;
	unsigned int genome_pos;
	unsigned int gap_left;		/* non-intron genome skips to left */
	unsigned int gap_right;		/* non-intron genome skips to right */
	char clone_base;
	char genome_base;
	double score;
};

struct SpaParameters
{
	unsigned int shortest_intron_length;
	unsigned int longest_intron_length;
	double score_match;
	double score_mismatch;
	double score_splice[256];
	double score_intron[100];
	double min_diff_misorientation;
	double score_genome_jump_beta0;
	double score_genome_jump_beta1;
	double score_genome_jump_beta2;
	double score_genome_jump_beta3;
	double score_genome_jump_alpha;
	double score_genome_extension;
	double score_clone_jump_beta0;
	double score_clone_jump_beta1;
	double score_clone_jump_beta2;
	double score_clone_jump_beta3;
	double score_clone_jump_alpha;
};

struct SpaTopParameters
{
	unsigned int tilesize;
	unsigned int tileoverlay;
        unsigned int maximum_locus_expand;
	unsigned int locus_expand;
        unsigned int clone_expand;
	unsigned int best_locus_list_size_max;
        unsigned int retile_poly_a_percentage;
	unsigned int define_radius;
        unsigned int retile_genomic_gap_lower_limit;
        unsigned int retile_genomic_gap_upper_limit;
	unsigned int retile_tilesize;
        unsigned int retile_tileoverlay;
        unsigned int retile_exon_extension;
	unsigned int init_cell_number_cutoff;
        unsigned int extension_cell_number_cutoff;
        unsigned int inside_retiling_cell_number_cutoff;
};

#ifdef __cplusplus
extern "C" {
#endif

/* error codes */
#define PARAM_SUCCESS				0
#define PARAM_ERROR				51
#define PARAM_OUTOFMEMORY_ERROR			52
#define PARAM_FILENAME_ERROR			53
#define PARAM_CORRUPTFILE_ERROR			54
#define PARAM_INVALIDPOINTER_ERROR		55
#define PARAM_INCONSISTANTPROBABILITY_ERROR	56
#define PARAM_INTRON_NUMBER_ERROR		57

/* setParamtersFromFile() - opens a file and parses it for parameters.
	All found parameters are set accordingly. Parameters remain
	set as long as the program is running. Parameters will have
	default values until set. Unknown parameters and improper file
	format yields error codes.
*/
int setParametersFromFile(char *filename);

/* getParametersFromFile() - opens a file and parses it for parameters.
	All found parameters are stored in the passed SpaParameters struct.
	Unknown parameters and improper file format yields error codes.
*/
int getParametersFromFile(struct SpaParameters *parameters, char *filename);

/* setParametersFromStruct() - sets program parameters from values in
	passed SpaParameters struct. Null pointer argument yields error code.
*/
int setParametersFromStruct(struct SpaParameters *parameters);

/* error codes */
#define LOCUS_SUCCESS				0
#define LOCUS_ERROR				21
#define LOCUS_OUTOFMEMORY_ERROR			22
#define LOCUS_TILESIZE_ERROR			23
#define LOCUS_TILEOVERLAY_ERROR			24
#define LOCUS_NIBFILENAME_ERROR			25
#define LOCUS_CORRUPTNIB_ERROR			26
#define LOCUS_LOCUSOFFSET_ERROR			27

/* createLocus() - indexes tiles from nibfile between start and end points
	the created locus is retuned in the first argument. The return value
	indicates success (0) or an error code. Constraints are:
		1 <= tileSize <= 15,
		1 <= tileOverlay <= tileSize
		locusStartOffset <= locusEndOffset < (# of bases in nib file)
			** note: locus offests are zero based indecies
	tileOverlay is the number of overlapping tiles at any position in the
	target sequence. tileOverlay = 1 means that tiles are strictly adjacent
	(no overlay) during the indexing of the target. 
	Note: the returned Locus is dynamically allocated; it is the caller's
	responsibility to free this memory when no longer needed by calling
	destroyLocus().
*/
int createLocus(struct Locus **createdLocus,
		char *targetNibFilename,
		unsigned int locusStartOffset,
		unsigned int locusEndOffset,
		unsigned int tileSize,
		unsigned int tileOverlay,
                unsigned int maximumLocusExpand,
                unsigned int locusExpand,
                unsigned int nibFileSize);

/* destroyLocus() - free the memory allocated to a Locus */
void destroyLocus(struct Locus **doomedLocus);

/* error codes */
#define GETDEFPOS_SUCCESS			0
#define GETDEFPOS_ERROR				31
#define GETDEFPOS_OUTOFMEMORY_ERROR		32
#define GETDEFPOS_EXTEND_INIT_ERROR		33
#define GETDEFPOS_QUERYSEQ_ERROR		34
#define GETDEFPOS_BAD_LOCUS_ERROR		35
#define DEFPOS_EMPTY				36

/* createDefPosList() - computes a linked list of diagSeg's and returns it in the
	first argument (defPosList). The return value indicates success (0) or
	an error code. Argument constraints are:
	NOTE: defPosList will be set to point to a dynamically allocated
	linked list. It is the caller's responsibilty to free this memory
	when no longer needed. Use destroyDefPosList().
	Call setExtendCriteria() before calling this function.
*/
int createDefPosList(struct diagSeg **defPosList,
		struct dnaSeq *querySequence,
		struct Locus *locus);

/* destroyDefPosList() - frees the memory of a linked list of diagSeg.
	Pass a pointer to the pointer to the start of the list. The
	list is freed, and the start pointer is set to NULL */
void destroyDefPosList(struct diagSeg **pList);

/* addDefPos() - add a single defined position to a defPosList */
int addDefPos(struct diagSeg **defPosList,
		unsigned int qOffset,
		unsigned int tOffset,
		unsigned int querySize);

/* normalizeDefPos() - make the retiled defined positions compatible with the previous ones */
void normalizeDefPos(struct diagSeg **in, 
                     unsigned int qSize, 
                     unsigned int sqStart, 
                     unsigned int sqEnd);

/* insertDefPos() - merge two lists of defined positions */
void insertDefPos(struct diagSeg **defPosList_in,
                  struct diagSeg **defPosList_out);

#define SPA_SUCCESS				0
#define SPA_ERROR				41
#define SPA_OUTOFMEMORY_ERROR			42
#define GRID_EMPTY				43
#define GRID_OVERLOADED				44
#define INCREASE_TILESIZE			45

#define SPA_CLONE_ORIENTATION_CORRECT		0
#define SPA_CLONE_ORIENTATION_UNKNOWN		2	
#define SPA_CLONE_ORIENTATION_MISORIENTED	1

#define TMAX 100
#define TMIN 4 

/* createAlignment() - determines the highest scoring alignment according to genome
	and query, along with internal jump scoring functions. Returns a pointer
	to the first Cell in a doubly linked list of aligned positions in the
	first argument. The return value indicates success (0) or an error code.
	NOTE: alignment_start will be set to point to a dynamically allocated
	linked list. It is the caller's responsibility to free this memory
	when no longer needed. Use destroyAlignment(). The alignment_score argument
	will be set by the function to equal the alignment score for the entire
	alignment (equal to the last cell's score). Assuming that the query
	has the proper orientation, then if it is on the negative strand, and
	therefore it has been reverse-complemented before being passed in to
	this function as an argument, you should pass a non-zero value (true)
	in to the negative_strand_rc argument, in order to score splice junctions
	according to the reverse complement of their frequency distribution on the
	positive strand. If the query has not been reverse-complemented, then pass
	zero (false) into negative_strand_rc. The misorientation argument functions
	as both in incoming argument and a return value argument. If the caller
	passes a pointer to an integer with either the value
	SPA_CLONE_ORIENTATION_CORRECT or SPA_CLONE_ORIENTATION_MISORIENTED,
	then the alignment will be done once using the assumed misorientation:
		negative_strand_rc	misorientation	splice junction scoring
		0			CORRECT		direct (gtag)
		0			MISORIENTED	reverse complement (ctac)
		!0			CORRECT		reverse complement (ctac)
		!0			MISORIENTED	direct (gtag)
	In these cases, the misorientation argument will not be changed. If the
	caller passes a pointer to the value SPA_CLONE_ORIENTATION_UNKNOWN, then
	the function will construct the alignment using both direct and reverse
	complemented splice junction scoring, and detect which produces the best
	alignment score. It will overwrite the value pointed to by orientation:
	if the query is misoriented (negative_strand_rc is zero and reverse
	complement splice junction scoring gives a better alignment score, or
	negative_strand_rc is non-zero and direct splice junction scoring gives
	a better alignment score) it will be set to
	SPA_CLONE_ORIENTATION_MISORIENTED. Otherwise, it will be set to
	SPA_CLONE_ORIENTATION_CORRECT.
*/
int createAlignment(struct Cell **alignment_start,
		double *alignment_score,
		struct dnaSeq *querySequence,
		struct Locus *locus,
		struct diagSeg *defPosList,
		int negative_strand_rc,
		int *misorientation,
                unsigned int cloneExpand);

void destroyAlignment(struct Cell **alignment_start);

unsigned int count_clump_hit(struct diagSeg *d);

/* c wrappers to needed Locus member functions */
const char *getLocusDna (struct Locus *locus);

char getLocusBase (struct Locus *locus, int offset);

#ifdef __cplusplus
}
#endif

/* end spa.h */
