/* gfClient - A client for the genomic finding program that produces a .psl file. */
/* Copyright 2001-2002 Jim Kent.  All rights reserved. */
#include "common.h"
#include "limits.h"
#include "linefile.h"
#include "aliType.h"
#include "fa.h"
#include "genoFind.h"
#include "psl.h"
#include "nib.h"
#include "exon.h"
#include "spa.h"

static void printAlignment   ( struct dnaSeq *qseq, struct Locus *tlocus,
                               struct Cell *alignment, char orientation, char *chrName, FILE *out, int misorientation);
static int processLoci       ( locusElem *bestLocus, char *bestStrand, unsigned int bestLocusListSize,
                               struct dnaSeq *seq, char *nibDir, FILE *out );
static int collectOutputData ( struct dnaSeq *qseq, struct Cell *alignment, struct Locus *tlocus, int *alignBeg, int *alignEnd,
                               char orientation, char **cloneAlignedSeq, char **chrAlignedSeq, int allocatedSeqSize, 
                               int *numExons, exonList *exons );
static int three_prime_tail_check(struct dnaSeq *qseq, int beg, int end, char orientation); 
static int seqGrow           ( char **seq, int newSeqSize );
static int seqsGrow          ( char **seqOne, char **seqTwo, int newSeqSize );
static int seqScore          ( char *seqA, char *seqB );

int maxInt ( int a, int b );

#define FALSE 0
#define TRUE 1

/*******************************************************************************
Parameter Default Values
*******************************************************************************/

static const unsigned int DEFAULT_TILESIZE = 12;
static const unsigned int MINIMUM_DEFAULT_TILESIZE = 4;
static const unsigned int DEFAULT_STEP_TILESIZE = 4;
static const unsigned int DEFAULT_STEP_START = 1;

/*******************************************************************************
Parameter declarations
*******************************************************************************/

static unsigned int tilesize = 12;
static unsigned int tilesize_step = 1;
static unsigned int tileoverlay = 2;
static unsigned int maximum_locus_expand = 50000;
static unsigned int locus_expand = 2000;
static unsigned int clone_expand = 20;
static unsigned int best_locus_list_size_max = 100;
unsigned int retile_poly_a_percentage = 100;  /* over this percentage,
                                        poly-a tails are not dynamically retiled */


unsigned int define_radius = 4; /* diagonal endpoint extra defined positions
                                        (uses manhattan distance around each endpoint */
unsigned int retile_genomic_gap_lower_limit = 3; /* gaps shorter than this
                                        will not be retiled */
unsigned int retile_genomic_gap_upper_limit = 200000; /* gaps longer than this
                                        will not be retiled */
unsigned int retile_tilesize = 8; /* dynamic retile tilesize */
unsigned int retile_tileoverlay = 2; /* dynamic retile tileoverlay */
unsigned int retile_exon_extension = 3; /* dynamic retile exon extension gap limit */
/* Control the initial number of defined positions  */
unsigned int init_cell_number_cutoff = 120000;
/* Control the number of defined positions occuring from locus extension */
unsigned int extension_cell_number_cutoff = 50000;
/* Control the number of defined positions occuring during retiling */
unsigned int inside_retiling_cell_number_cutoff = 25000;

/*******************************************************************************
Parameter File Parsing Code
*******************************************************************************/

static void getLineFromFile(char *buffer, unsigned int buffer_size, FILE *fin)
{
	int cbuf;
	while (buffer_size > 1) {
		cbuf = fgetc(fin);
		if (cbuf == EOF || cbuf == '\n') {
			*buffer = '\0';
			return;
		}
		*buffer++ = (char)cbuf;
		buffer_size--;
	}
	*buffer = '\0';
	/* throw away any extra characters on long line */
	while (!feof(fin) && fgetc(fin) != '\n');
}

int getTopParameter(struct SpaTopParameters *parameters, char *pname, char *pvalue)
{
	static const char *parametername[] = {
		"tilesize",
		"tileoverlay",
                "maximum_locus_expand",
		"locus_expand",
                "clone_expand",
		"best_locus_list_size_max",
		"define_radius",
		"init_cell_number_cutoff",
		"extension_cell_number_cutoff",
		"inside_retiling_cell_number_cutoff",
		"retile_genomic_gap_lower_limit",
		"retile_genomic_gap_upper_limit",
		"retile_poly_a_percentage",
		"retile_tilesize",
		"retile_tileoverlay",
		"retile_exon_extension"
	};
	int number_of_parameters = sizeof(parametername) / sizeof(parametername[0]);
	int p_index;
	for (p_index = 0; p_index < number_of_parameters; p_index++) {
		if (strcmp(pname,parametername[p_index]) == 0) {
			unsigned int int_value;
			int result;
			result = sscanf(pvalue,"%u",&int_value);
			if (result != 1) return FALSE; /* bad value */
			switch (p_index) {
			case 0: parameters->tilesize = int_value; return TRUE;
			case 1: parameters->tileoverlay = int_value; return TRUE;
                        case 2: parameters->maximum_locus_expand = int_value; return TRUE;
			case 3: parameters->locus_expand = int_value; return TRUE;
                        case 4: parameters->clone_expand = int_value; return TRUE;
			case 5: parameters->best_locus_list_size_max = int_value; return TRUE;
			case 6: parameters->define_radius = int_value; return TRUE;
			case 7: parameters->init_cell_number_cutoff = int_value; return TRUE;
			case 8: parameters->extension_cell_number_cutoff = int_value; return TRUE;
			case 9: parameters->inside_retiling_cell_number_cutoff = int_value; return TRUE;
			case 10: parameters->retile_genomic_gap_lower_limit = int_value; return TRUE;
			case 11: parameters->retile_genomic_gap_upper_limit = int_value; return TRUE;
			case 12: parameters->retile_poly_a_percentage = int_value; return TRUE;
			case 13: parameters->retile_tilesize = int_value; return TRUE;
			case 14: parameters->retile_tileoverlay = int_value; return TRUE;
			case 15: parameters->retile_exon_extension = int_value; return TRUE;
			}
			return FALSE; /* logical error */
		}
	}
	/* could not find parameter name in list */
	return FALSE;
}

int getTopParametersFromFile(struct SpaTopParameters *parameters, char *filename)
{
	FILE *fin;
	if (parameters == 0) return PARAM_INVALIDPOINTER_ERROR;
	/* set parameters to defaults */
	/* read parameter values from file */
	if (filename == 0) return PARAM_FILENAME_ERROR;
	fin = fopen(filename, "r");
	if (!fin) return PARAM_FILENAME_ERROR;
	while (!feof(fin)) {
		char pline[512];
		char pname[256];
		char pvalue[256];
		int result;
		getLineFromFile(pline,512,fin);
		result = sscanf(pline,"%255s%255s",pname,pvalue);
		if (pname[0] == '#') continue; /* ignore comment lines */
		if (result == 0) continue; /* ignore blank lines */
		if (result == 1 || getTopParameter(parameters,pname,pvalue) == FALSE) {
			fprintf(stderr,"Error while parsing parameter file on line:\n%s\n",pline);
			return PARAM_CORRUPTFILE_ERROR;
		}
	}
	fclose(fin);
	return PARAM_SUCCESS;
}

int setTopParametersFromStruct(struct SpaTopParameters *parameters)
{
	if (parameters == 0) return PARAM_INVALIDPOINTER_ERROR;
	if (parameters->tilesize > TMAX || parameters->tilesize < TMIN) return LOCUS_TILESIZE_ERROR;
	if (parameters->tileoverlay > parameters->tilesize || parameters->tileoverlay < 1) 
		return LOCUS_TILESIZE_ERROR;
	tilesize = parameters->tilesize;
	if (parameters->tileoverlay > tilesize || parameters->tileoverlay < 1) 
		return PARAM_ERROR;
	tileoverlay = parameters->tileoverlay;
	if (parameters->maximum_locus_expand < 1) return PARAM_ERROR;
        maximum_locus_expand = parameters->maximum_locus_expand;
	if (parameters->locus_expand > maximum_locus_expand) return PARAM_ERROR;
	locus_expand = parameters->locus_expand;
	if (parameters->clone_expand < 1) return PARAM_ERROR;
        clone_expand = parameters->clone_expand;
	if (parameters->best_locus_list_size_max < 1) return PARAM_ERROR;
	best_locus_list_size_max = parameters->best_locus_list_size_max;
	if (parameters->define_radius < 1 || parameters->define_radius > tilesize) return PARAM_ERROR;
	define_radius = parameters->define_radius;
	if (parameters->init_cell_number_cutoff < 1) return PARAM_ERROR;
	init_cell_number_cutoff = parameters->init_cell_number_cutoff;
	if (parameters->extension_cell_number_cutoff < 0) return PARAM_ERROR;
	extension_cell_number_cutoff = parameters->extension_cell_number_cutoff;
	if (parameters->inside_retiling_cell_number_cutoff < 0) return PARAM_ERROR;
	inside_retiling_cell_number_cutoff = parameters->inside_retiling_cell_number_cutoff;
	if (parameters->retile_genomic_gap_lower_limit < 0) return PARAM_ERROR;
	retile_genomic_gap_lower_limit = parameters->retile_genomic_gap_lower_limit;
	if (parameters->retile_genomic_gap_upper_limit < 0) return PARAM_ERROR;
	retile_genomic_gap_upper_limit = parameters->retile_genomic_gap_upper_limit;
	if (parameters->retile_poly_a_percentage < 0 || parameters->retile_poly_a_percentage > 100)
		return PARAM_ERROR;
	retile_poly_a_percentage = parameters->retile_poly_a_percentage;
	if (parameters->retile_tilesize < TMIN || parameters->retile_tilesize > TMAX) return PARAM_ERROR;
	retile_tilesize = parameters->retile_tilesize;
	if (parameters->retile_tileoverlay > retile_tilesize || parameters->retile_tileoverlay < 1)
		return LOCUS_TILESIZE_ERROR;
	retile_tileoverlay = parameters->retile_tileoverlay;
	if (parameters->retile_exon_extension < 0) return PARAM_ERROR;
	retile_exon_extension = parameters->retile_exon_extension; 
	return PARAM_SUCCESS;
}

int setTopParametersFromFile(char *filename)
{
	struct SpaTopParameters parameters;
	int result = getTopParametersFromFile(&parameters, filename);
	if (result != PARAM_SUCCESS) return result;
	result = setTopParametersFromStruct(&parameters);
	if (result != PARAM_SUCCESS) return result;
	return PARAM_SUCCESS;
}

#define SHOW_TIMING
#ifdef SHOW_TIMING
#include <sys/times.h>
struct tms begintime;
struct tms endtime;
#endif

void usage()
/* Explain usage and exit. */
{
printf(
  "usage:\n"
  "   spa <host> <port> <spa_top_parameters file> <spa_parameters file> <nibDir> <in.fa> <out.mach>\n"
  "where\n"
  "   host is the name of the machine running the gfServer\n"
  "   port is the same as you started the gfServer with\n"
  "   spa_top_parameters is a parameter file which affects the search process\n"
  "   spa_parameters is another parameter file which affects the scoring scheme\n"
  "   nibDir is the path of the nib files needed by the server\n"
  "   in.fa a fasta format file.  May contain multiple records\n"
  "   out.mach where to put the output\n");
}


void insertIntoBestLoci(locusElem *bestLocus, char *bestStrand, unsigned int *bestLocusListSize,
		locusElem *locus, char strand)
{
  unsigned int i;

  locusElem *scan;
  for (scan = locus; scan != NULL; scan=scan->next) {
    if (*bestLocusListSize == best_locus_list_size_max &&
		scan->score <= bestLocus[*bestLocusListSize-1].score) continue;
    for (i = *bestLocusListSize; i>0; i--) {
      if (bestLocus[i-1].score > scan->score) {
        bestLocus[i].start = scan->start;
        bestLocus[i].end = scan->end;
        bestLocus[i].score = scan->score;
        strcpy(bestLocus[i].chrName, scan->chrName);
        bestStrand[i] = strand;
        break;
      }
      if (i < best_locus_list_size_max) {
        bestLocus[i].start = bestLocus[i-1].start;
        bestLocus[i].end = bestLocus[i-1].end;
        bestLocus[i].score = bestLocus[i-1].score;
        strcpy(bestLocus[i].chrName, bestLocus[i-1].chrName);
        bestStrand[i] = bestStrand[i-1];
      }
    }
    if (*bestLocusListSize < best_locus_list_size_max) (*bestLocusListSize)++;
    if (i == 0) {
      bestLocus[0].start = scan->start;
      bestLocus[0].end = scan->end;
      bestLocus[0].score = scan->score;
      strcpy(bestLocus[0].chrName, scan->chrName);
      bestStrand[0] = strand;
    }
  }
}

void lineFileFree(struct lineFile *lf) {
     struct lineFile *t, *next;
     for (t = lf; t != 0; t = next) {
         next = t->next;
         free(t->fileName);
         free(t->buf);
         free(t);
     }
}

void findBestLoci(locusElem *bestLocus, char *bestStrand, unsigned int *bestLocusListSize,
                locusElem *locus, locusElem *locusRC, int qSize)
{
  int i;
  *bestLocusListSize = 0;
  if (locus) insertIntoBestLoci(bestLocus, bestStrand, bestLocusListSize, locus, '+');
  if (locusRC) insertIntoBestLoci(bestLocus, bestStrand, bestLocusListSize, locusRC, '-');
                                                                                                                                           
  int eps1 = (bestLocus[0].score * 95) / 100;
  int eps2 = (bestLocus[0].score * 80) / 100;
  int looks_like_pseudo;
  if((bestLocus[0].end - bestLocus[0].start + 1) <= (3*qSize/2)){
    looks_like_pseudo = 1;
  }
  else{
    looks_like_pseudo = 0;
  }
                                                                                                                                           
  bestLocus[0].dothis = 1;
  for (i = 1; i < *bestLocusListSize; i++) {
    //test if we want to do this locus
    if(bestLocus[i].score >= eps1){
      bestLocus[i].dothis = 1;
    }
    else if(looks_like_pseudo && ((bestLocus[i].end-bestLocus[i].start+1) > (3*qSize/2)) && (bestLocus[i].score >= eps2)){
      bestLocus[i].dothis = 1;
    }
    else{
      bestLocus[i].dothis = 0;
    }
    if ((bestLocus[i].score < eps2) || (i > 19))
      *bestLocusListSize = i;
  }
}


/* Paramters to gfAlignStrand */
struct gfOutput *gvo;
const int minScore = 30;
const double minIdentity = 90;
char *outputFormat = "psl";


/* gfClient - A client for the genomic finding program that produces a .psl file. */
void gfClientHack (char *hostName, char *portName, char *nibDir, char *inName, 
		   char *outName, char *tTypeName, char *qTypeName) {
  struct lineFile *lf = lineFileOpen(inName, TRUE);
  struct dnaSeq seq;
  /*  struct ssBundle *bundleList;*/

  /* 20030227 */
  FILE *out = mustOpen(outName, "w");
  enum gfType qType = gfTypeFromName(qTypeName);
  enum gfType tType = gfTypeFromName(tTypeName);
  /*  int dotMod = 0;*/
  /**/

  char databaseName[256];
  int databaseNamepos;
  char *scan;
  locusElem *bestLocus; /* holds the best loci from blat alignments */
  char *bestStrand; /* holds the corresponding senses for bestLocus */

  locusList loci;
  locusList lociRC;

  databaseNamepos = 0;
  scan = hostName;
  while (databaseNamepos < 255 && *scan) databaseName[databaseNamepos++] = *scan++;
  if (databaseNamepos < 255) databaseName[databaseNamepos++] = ':';
  scan = portName;
  while (databaseNamepos < 255 && *scan) databaseName[databaseNamepos++] = *scan++;
  databaseName[databaseNamepos] = '\0';

  /* 20030227 */
  gvo = gfOutputAny(outputFormat,  round(minIdentity*10), qType == gftProt, tType == gftProt,
		    FALSE, databaseName, 23, 3.0e9, out);
  /**/
  
  fprintf (out, "#clone_id\tclone_seq\tclone_length\tchromosome_id\tchromosome_start\tchromosome_end\tchromosome_length\tchromosome_seq\tclone_orientation\tclone_score\tnumber_of_exons\tmisorientation\n");
  
  fprintf (out, "#exon_id\tclone_id\texon_start\texon_end\texon_seq\tchromosome_start\tchromosome_end\tchromosome_seq\texon_orientation\texon_score\n");
  
  /* Here is where the alignment is generated, sequence by sequence */
  /*  while (faSomeSpeedReadNext(lf, &seq.dna, &seq.size, &seq.name, TRUE)) { */
  bestLocus = (locusElem *)malloc(best_locus_list_size_max * sizeof(locusElem));
  bestStrand = (char *)malloc(best_locus_list_size_max * sizeof(char));
  while (faSpeedReadNext(lf, &seq.dna, &seq.size, &seq.name)) {
    unsigned int bestLocusListSize;
    int conn = gfConnect(hostName, portName);
    locusInit ( &loci );
    gfAlignStrandHack(&conn, nibDir, &seq, FALSE, minScore, gvo, &loci, NULL);
    conn = gfConnect(hostName, portName);
    locusInit ( &lociRC );
    reverseComplement(seq.dna, seq.size);
    gfAlignStrandHack(&conn, nibDir, &seq, TRUE,  minScore, gvo, &lociRC, NULL);
    reverseComplement(seq.dna, seq.size);
    findBestLoci(bestLocus, bestStrand, &bestLocusListSize, loci.head, lociRC.head, seq.size);
    processLoci(bestLocus, bestStrand, bestLocusListSize, &seq, nibDir, out);
    locusFree (&loci);
    locusFree (&lociRC); 
  }
  lineFileFree(lf);
  free(gvo->data);
  free(gvo);
  fclose(out);
  free(bestStrand);
  free(bestLocus);
  /* The following func frees the speed read stuff */
  faFreeFastBuf();

}

char * basedir (const char *name) {
      const char *base;
      char *dir;
      int i = 0;

      for (base = (name+i); *(name+i); i++) {
           if (*(name+i) == '/') {
	        base = name + i + 1;
	   }
      }
      dir = (char *) malloc((strlen(name)-strlen(base)+1)*sizeof(char));
      strncpy(dir, name, strlen(name)-strlen(base));
      return dir;
}

/* Process command line. */
int main(int argc, char *argv[]) {
  int retval;
  char *tType = "dna";
  char *qType = "dna";

  if (argc != 8) {
    usage();
    return SPA_ERROR;
  }
  
  retval = setTopParametersFromFile(argv[3]);
  if (retval != PARAM_SUCCESS) {
    fprintf(stderr, "Error: setTopParametersFromFile() returned error code %d\n", retval);
    return retval;
  }

  retval = setParametersFromFile(argv[4]);
  if (retval != PARAM_SUCCESS) {
    fprintf(stderr, "Error: setParametersFromFile() returned error code %d\n", retval);
    return retval;
  }
  gfClientHack (argv[1], argv[2], argv[5], argv[6], argv[7], tType, qType);

  return 0;
}


void printAlignment  ( struct dnaSeq *qseq, struct Locus *tlocus,
                      struct Cell *alignment, char orientation, char *chrName, FILE *out, int misorient) {
  
  int alignBeg = 0;
  int alignEnd = 0;
  int allocatedSeqSize;
  char *cloneAlignedSeq;
  char *chrAlignedSeq;
  int numExons = 0;
  exonList exons;
  exonElem *exon;
  int exonIdx = 1;

  /* start building chromosome and clone seqs with -'s */
  allocatedSeqSize = qseq->size+1;
  if ( ( cloneAlignedSeq = (char *)malloc(allocatedSeqSize) ) == NULL ) {
    printf ("ERROR No space to malloc cloneAlignedSeq!\n");
    exit(1);
  }
  if ( ( chrAlignedSeq = (char *)malloc(allocatedSeqSize) ) == NULL ) {
    printf ("ERROR No space to malloc chrAlignedSeq!\n");
    exit(1);
  }
  exonInit ( &exons );
  exon = exons.head;
  /*collect output data */
  collectOutputData ( qseq, alignment, tlocus, &alignBeg, &alignEnd, orientation,
		      &cloneAlignedSeq, &chrAlignedSeq, allocatedSeqSize, &numExons,
		      &exons );
  fprintf (out, "%s\t%s\t%i\t%s\t%i\t%i\t%i\t%s\t%c\t%i\t%i\t%i\n",
	   qseq->name, cloneAlignedSeq, qseq->size, chrName, alignBeg+1, (alignEnd+1), alignEnd-alignBeg+1, 
	   chrAlignedSeq, orientation, seqScore ( cloneAlignedSeq, chrAlignedSeq ), numExons, misorient);


  exon = exons.head;

  while ( exon != NULL ) {

    fprintf (out, "%s.%i\t%s\t%i\t%i\t%s\t%i\t%i\t%s\t%c\t%i\n",
	     qseq->name,
	     exonIdx,
	     qseq->name,
	     exon->clone_beg+1,
	     exon->clone_end+1,
	     exon->clone_seq,
	     exon->genome_beg+1,
	     exon->genome_end+1,
	     exon->genome_seq,
	     orientation,
	     exon->score );
    exon=exon->next;
    exonIdx++;
  }

  fprintf (out, "//\n");

  exonFree ( &exons );

  if ( cloneAlignedSeq != NULL ) 
    free(cloneAlignedSeq);
  if ( chrAlignedSeq != NULL )
    free(chrAlignedSeq);
}


void destroySpaAlignment(struct Locus **tlocus, struct Cell **alignment)
{
  destroyLocus(tlocus);
  destroyAlignment(alignment);
}


int createSpaAlignment(struct Locus **tlocus, struct diagSeg **defPosList, struct Cell **alignment,
                       int *misorientation, double *alignmentScore,
                       struct dnaSeq *qseq, char *targetNibFileName, 
		       unsigned int locusStart, unsigned int locusEnd, unsigned int maximumLocusExpand,
                       unsigned int locusExpand, unsigned int cloneExpand, unsigned int nibFileSize, char orientation) {
  
  int retval;
  unsigned int tSize = tilesize + tilesize_step - 1;

  /* get actual alignment */  
  do {
     retval = createLocus(tlocus, targetNibFileName, locusStart, locusEnd, tSize, tileoverlay,
                       maximumLocusExpand, locusExpand, nibFileSize);
     if (retval != LOCUS_SUCCESS) {
         destroyLocus(tlocus);
         fprintf(stderr, "Error : createLocus() returned error code %d\n", retval);
         return retval;
     }
   
     retval = createDefPosList(defPosList, qseq, *tlocus);    
     if (retval == DEFPOS_EMPTY) {
         destroyLocus(tlocus);
         tSize = tSize - DEFAULT_STEP_TILESIZE;
     }
     else
         break;
  } while (tSize >= MINIMUM_DEFAULT_TILESIZE);
  if (retval != SPA_SUCCESS) {
      if (retval != DEFPOS_EMPTY) {
          fprintf(stderr, "Warning : getDefPosList() returned error code %d\n", retval);
      }
      destroyLocus(tlocus);
      return retval;
  }
#undef SHOW_RETURNED_DEFPOS
#ifdef SHOW_RETURNED_DEFPOS
int tcount=0;
int qcount=0;
struct diagSeg *scan;
for (scan = *defPosList; scan; scan = scan->next){
printf("seg: (%u %u) - (%u %u)\n",scan->qStart,scan->qEnd,scan->tStart,scan->tEnd);
unsigned t;
for (t=scan->tStart; t<= scan->tEnd; t++)
printf("%c",getLocusBase(*tlocus,t));
printf("\n");
tcount += scan->tEnd - scan->tStart;tcount++;
qcount += scan->qEnd - scan->qStart;qcount++;
}
printf("totals: %d - %d\n",qcount,tcount);
#endif
  *misorientation = SPA_CLONE_ORIENTATION_UNKNOWN;
  retval = createAlignment(alignment, alignmentScore,
			qseq, *tlocus, *defPosList,
			orientation - '+', misorientation, cloneExpand);
  if (retval != SPA_SUCCESS) {
      if ((retval == GRID_OVERLOADED) && (tSize <= MINIMUM_DEFAULT_TILESIZE)) {
           retval = DEFPOS_EMPTY;
      }
      if ((retval != GRID_OVERLOADED) && 
          (retval != INCREASE_TILESIZE) && 
          (retval != GRID_EMPTY) && (retval != DEFPOS_EMPTY)) {
           fprintf(stderr, "Error : createAlignment() returned error code %d\n", retval);
      }
      return retval;
  }
  return 0;
}


int processLoci ( locusElem *bestLocus, char *bestStrand, unsigned int bestLocusListSize,
		struct dnaSeq *seq, char *nibDir, FILE *out ) {

  unsigned int nibFileSize;
  FILE *nib = NULL;
  char nibFileName[512];
  locusElem *locus_scan;
  /* next three arrays hold results from spa -- so that scores can be compared */
  struct Locus **locus = (struct Locus **)malloc(best_locus_list_size_max * sizeof(struct Locus *));
  struct diagSeg **defPosList = (struct diagSeg **)malloc(best_locus_list_size_max * sizeof(struct diagSeg *));
  struct Cell **alignment = (struct Cell **)malloc(best_locus_list_size_max * sizeof(struct Cell *));
  double *alignmentScore = (double *)malloc(best_locus_list_size_max * sizeof(double));
  int *misorientation = (int *)malloc(best_locus_list_size_max * sizeof(int));
  int best_spa_index = 0;
  int i;

#ifdef SHOW_TIMING
times(&begintime);
#endif
  int retval;
  int valid[bestLocusListSize];
  int num_loci_done = 0;  
  
  for ( i=0; i<bestLocusListSize; i++) {
        valid[i] = 0;
  }
  for ( i=0; i<bestLocusListSize; i++) {
    if (bestLocus[i].dothis == 1) {
        ++num_loci_done;
        /* Test for perfect match and add extra bases at the sides */
            
        char orientation;
        locus_scan = &bestLocus[i];
        orientation = bestStrand[i];
        sprintf(nibFileName, "%s/%s%s", nibDir, locus_scan->chrName, ".nib");
        nibOpenVerify (nibFileName, &nib, &nibFileSize );
        fclose(nib);
        if (orientation == '-') reverseComplement(seq->dna,seq->size);
    	retval = createSpaAlignment( &locus[i], &defPosList[i], &alignment[i], /* these need to be destroyed later */
        	 &misorientation[i], &alignmentScore[i], seq, nibFileName, locus_scan->start,
            	locus_scan->end, maximum_locus_expand, locus_expand, clone_expand, nibFileSize, orientation);
        if (orientation == '-') reverseComplement(seq->dna, seq->size);
    	if ((retval == INCREASE_TILESIZE) || (retval == GRID_OVERLOADED)) {
        	tilesize_step += DEFAULT_STEP_TILESIZE; 
        	destroySpaAlignment(&locus[i],&alignment[i]);
        	i--;
		num_loci_done--;
    	} 
    	else if ((retval != DEFPOS_EMPTY) && (retval != LOCUS_TILESIZE_ERROR)) {
        	valid[i] = 1; 
        	tilesize_step = DEFAULT_STEP_START;
        	if (alignmentScore[i] > alignmentScore[best_spa_index]) best_spa_index = i;
    	}
    	else if (retval == LOCUS_TILESIZE_ERROR) {
                 valid[i] = 0;
        	 i = bestLocusListSize;
         	tilesize_step = DEFAULT_STEP_START;
     	}
      }
    }
    if (num_loci_done > 0) { 
	if (bestStrand[best_spa_index] == '-')
		reverseComplement(seq->dna,seq->size);
     	if (misorientation[best_spa_index] == 1) {
		if (bestStrand[best_spa_index] == '-') 
			bestStrand[best_spa_index] = '+';
		else
			bestStrand[best_spa_index] = '-';
	} 
    if  (alignment[best_spa_index] != NULL) {
        printAlignment  ( seq, locus[best_spa_index], 
        alignment[best_spa_index], bestStrand[best_spa_index], bestLocus[best_spa_index].chrName, out, misorientation[best_spa_index]);
    }
  }
  for ( i=0; i< bestLocusListSize; i++) {
        if (valid[i] == 1) {
            destroySpaAlignment(&locus[i],&alignment[i]);
        }
  }
#ifdef SHOW_TIMING
times(&endtime);
printf("Time : %.2f seconds processing %d loci for sequence: %s\n",(double)(endtime.tms_utime - begintime.tms_utime)/sysconf(_SC_CLK_TCK), num_loci_done, seq->name); fflush(0);
#endif
  free(misorientation);
  free(alignmentScore);
  free(alignment);
  free(defPosList);
  free(locus);
  return 0;
}

static int three_prime_tail_check(struct dnaSeq *qseq, int beg, int end, char orientation) {
          int count = 0;
          int gap = end - beg + 1; 
          int pos;

          if (orientation == '+') {
              for (pos=beg; pos <= end; pos++){
                   char base = qseq->dna[pos];
                   if (base == 'a' || base == 'A') count++;
              }
          } else {
              for (pos=beg; pos <= end; pos++){
                   char base = qseq->dna[pos];
                   if (base == 't' || base == 'T') count++;
              }
          } 
          if (count * 100 > retile_poly_a_percentage * gap)
              return 1;
          return 0;
} 


static int collectOutputData ( struct dnaSeq *qseq, struct Cell *alignment, struct Locus *tlocus, int *alignBeg, int *alignEnd,
                               char orientation, char **cloneAlignedSeq, char **chrAlignedSeq, int allocatedSeqSize,
                               int *numExons, exonList *exons ) {
                                                                                                                                                             
  struct Cell *alignmentScan;
  struct Cell *alignCoords;
  int prevClone = -1;
  int prevChr = -1;
  int reallocWindow = 512;
  int seqSize = 0;
  int i = 0;
  int retval = 0;
  int minIntronLen = 30;
  int jumpSize;
  exonElem *exon;
  int cloneExonBeg;
  int chrExonBeg;
  int cloneExonEnd = -1;
  int chrExonEnd = -1;
  char *cloneExonSeq;
  char *chrExonSeq;
  int exonSeqSize = 0;
  int skipCloneIdx = 0;
  int exonAllocatedSeqSize = reallocWindow;
  int cloneEnd;
  int coordTemp = 0;

  char deletionChar = '-';

  exon = exons->head;
                                                                                                                                                             
  if ( ( cloneExonSeq = (char *)malloc(exonAllocatedSeqSize) ) == NULL ) {
    printf ("ERROR No space to malloc cloneExonSeq!\n");
    return 1;
  }
  if ( ( chrExonSeq = (char *)malloc(exonAllocatedSeqSize) ) == NULL ) {
    printf ("ERROR No space to malloc chrExonSeq!\n");
    return 1;
  }
                                                                                                                                                             
                                                                                                                                                             
  /************************************************************************/
  /* retrieve the locus coords... */
  alignCoords = alignment;
  *alignBeg = alignCoords->genome_pos;
  cloneExonBeg = alignCoords->clone_pos;
  chrExonBeg = alignCoords->genome_pos;
  //printf("%d %d %d\n", *alignBeg, cloneExonBeg, chrExonBeg);
                                                                                                                                                             
  while ( alignCoords->child != 0 )
    alignCoords = alignCoords->child;
  *alignEnd = alignCoords->genome_pos;
  /* cloneEnd = alignCoords->clone_pos; */
  /**/
                                                                                                                                                             
  cloneEnd = qseq->size - 1;
                                                                                                                                                             
  /* collect info */
  alignmentScan = alignment;
  if (alignmentScan != 0) {
    prevChr = alignmentScan->genome_pos-1;
    prevClone = -1;
    if ((orientation == '-') && (alignmentScan->clone_pos != 0)) {
         if (three_prime_tail_check(qseq, 0, alignmentScan->clone_pos - 1, orientation)) 
	     deletionChar = '*';
             /* prevClone = alignmentScan->clone_pos - 1; */
    }
  }
  (*cloneAlignedSeq)[seqSize] = '\0';
  (*chrAlignedSeq)[seqSize] = '\0';
                                                                                                                                                             
  /* gap handling */
  while (alignmentScan != 0) {
                                                                                                                                                             
    /*
      printf ("W %i %i %i %i %c %c\n", alignmentScan->clone_pos,
      alignmentScan->genome_pos, alignmentScan->gap_left, alignmentScan->gap_right,
      alignmentScan->clone_base, alignmentScan->genome_base
      );
    */
    /*    alignmentScan = alignmentScan->child;*/
                                                                                                                                                             
    /*    continue;*/
                                                                                                                                                             
    if ( ( seqSize + 2 ) >= allocatedSeqSize ) {
      /*      printf ("Reallocating from %i to %i seq is %i\n", allocatedSeqSize, allocatedSeqSize+reallocWindow, seqSize);*/
      allocatedSeqSize += maxInt(reallocWindow, 2);
      seqsGrow ( cloneAlignedSeq, chrAlignedSeq, allocatedSeqSize );
    }
                                                                                                                                                             
    /*uuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuu*/
    /* if there's a gap in the genome string it may be an intron...*/
    if ( prevChr != alignmentScan->genome_pos-1 ) {
        /*
        printf ("GAP %i %c %i %c\n",
        alignmentScan->parent->clone_pos,
        alignmentScan->parent->clone_base,
        alignmentScan->clone_pos,
        alignmentScan->clone_base);
                                                                                                                                                             
            printf ("%i %i %i %i\n", alignmentScan->parent->clone_pos,
            alignmentScan->parent->genome_pos,
            alignmentScan->parent->gap_left, alignmentScan->parent->gap_right);
            printf ("%i %i %i %i\n\n", alignmentScan->clone_pos,
            alignmentScan->genome_pos, alignmentScan->gap_left, alignmentScan->gap_right);
        */
      jumpSize = alignmentScan->genome_pos-1 - prevChr;
            //printf ("%i %i %i JJJ\n", alignmentScan->genome_pos-1, prevChr, jumpSize);
      /* if gap is smaller than min intron length, it can't be an intron... */
      /*if ( jumpSize < minIntronLen ) { */
      /* jumpSize - parent->gap_right - gap_left must be longer than min intron len to be intron...*/
      //printf("JumpSize : %d\n", jumpSize);
      //printf("gap_right : %d\n", alignmentScan->parent->gap_right);
      //printf("gap_left  : %d\n", alignmentScan->gap_left);
      //printf("minIntron : %d\n", minIntronLen);
      int test = jumpSize-alignmentScan->parent->gap_right-alignmentScan->gap_left;
      if (test < minIntronLen) {
        if ( ( seqSize + jumpSize + 2 ) >= allocatedSeqSize ) {
          allocatedSeqSize += maxInt(reallocWindow, jumpSize+1);
          seqsGrow ( cloneAlignedSeq, chrAlignedSeq, allocatedSeqSize );
        }
                                                                                                                                                             
        if ( ( exonSeqSize + jumpSize + 2 ) >= exonAllocatedSeqSize ) {
          exonAllocatedSeqSize += maxInt(reallocWindow, jumpSize+1);
          seqsGrow ( &cloneExonSeq, &chrExonSeq, exonAllocatedSeqSize );
        }
                                                                                                                                                             
        while ( prevChr < alignmentScan->genome_pos-1 ) {
          prevChr++;
          (*chrAlignedSeq)[seqSize] = getLocusBase ( tlocus, prevChr );
          (*cloneAlignedSeq)[seqSize] = '-';
          seqSize++;
          chrExonSeq[exonSeqSize] = getLocusBase ( tlocus, prevChr );
          cloneExonSeq[exonSeqSize] = '-';
          exonSeqSize++;
        }
        //printf("seq1 : %s\n", *chrAlignedSeq);
        //printf("seq2 : %s\n", *cloneAlignedSeq);
      }
                                                                                                                                                             
      else {

        /* if it can be an intron */
        if ( ( seqSize + 11 ) >= allocatedSeqSize ) {
          /*      printf ("Reallocating from %i to %i seq is %i\n", allocatedSeqSize, allocatedSeqSize+reallocWindow, seqSize);*/
          allocatedSeqSize += maxInt(reallocWindow, 11);
          seqsGrow ( cloneAlignedSeq, chrAlignedSeq, allocatedSeqSize );
        }
                                                                                                                                                             
        /* additions in clone before splice junction */
        /*      printf ("%i %i\n", prevChr, alignmentScan->genome_pos-1 );*/
        for ( prevChr = alignmentScan->parent->genome_pos + 1;
              prevChr <= alignmentScan->parent->genome_pos + alignmentScan->parent->gap_right;
              prevChr++ ) {
                                                                                                                                                             
          if ( ( seqSize + 12 ) >= allocatedSeqSize ) {
            /*      printf ("Reallocating from %i to %i seq is %i\n", allocatedSeqSize, allocatedSeqSize+reallocWindow, seqSize);*/
            allocatedSeqSize += maxInt(reallocWindow, 12);
            seqsGrow ( cloneAlignedSeq, chrAlignedSeq, allocatedSeqSize );
          }
                                                                                                                                                             
          if ( ( exonSeqSize + 2 ) >= exonAllocatedSeqSize ) {
            exonAllocatedSeqSize += maxInt(reallocWindow, 2);
            seqsGrow ( &cloneExonSeq, &chrExonSeq, exonAllocatedSeqSize );
          }
                                                                                                                                                             
          (*chrAlignedSeq)[seqSize] = getLocusBase ( tlocus, prevChr );
          (*cloneAlignedSeq)[seqSize] = '-';
          seqSize++;
          chrExonSeq[exonSeqSize] = getLocusBase ( tlocus, prevChr );
          cloneExonSeq[exonSeqSize] = '-';
          exonSeqSize++;
        } /*end pre-splice junction gap*/
                                                                                                                                                             
        chrExonSeq[exonSeqSize] = '\0';
        cloneExonSeq[exonSeqSize] = '\0';
                                                                                                                                                             
        /*      printf ( "Exon:\n%i-%i\n%s\n%i-%i\n%s\n", chrExonBeg, chrExonEnd, chrExonSeq, cloneExonBeg, cloneExonEnd, cloneExonSeq );*/
                                                                                                                                                             
        /************************************************************************/
        /* create new exon member                                               */
        /************************************************************************/
                                                                                                                                                             
        /* the next 2 for loops widen the genome coord range of the ends of the exon has any */
        /* deletions in the clone*/
        for ( skipCloneIdx = 0; (cloneExonSeq[skipCloneIdx] == '-' || chrExonSeq[skipCloneIdx] == '-' || chrExonSeq[skipCloneIdx] == '*'); skipCloneIdx++ ) {
              if (cloneExonSeq[skipCloneIdx] == '-')
                  chrExonBeg--;
              else
                  cloneExonBeg--;
        }
        for ( skipCloneIdx = exonSeqSize-1; (cloneExonSeq[skipCloneIdx] == '-' || chrExonSeq[skipCloneIdx] == '-'); skipCloneIdx-- ) {
              if (cloneExonSeq[skipCloneIdx] == '-')
                  chrExonEnd++;
              else
                  cloneExonEnd++;
        }
        if ( orientation == '-' ) {
          coordTemp = cloneEnd - cloneExonBeg;
          cloneExonBeg = cloneEnd - cloneExonEnd;
          cloneExonEnd = coordTemp;
                                                                                                                                                             
          reverseComplement ( cloneExonSeq, strlen(cloneExonSeq) );
          reverseComplement ( chrExonSeq, strlen(chrExonSeq) );
          exonAddBegMalloc ( exons, cloneExonBeg, cloneExonEnd,
                             chrExonEnd, chrExonBeg, cloneExonSeq, chrExonSeq,
                             seqScore(cloneExonSeq, chrExonSeq) );
                                                                                                                                                             
        }
        else {
          exonAddMalloc ( exons, exons->tail, cloneExonBeg, cloneExonEnd,
                          chrExonBeg, chrExonEnd, cloneExonSeq, chrExonSeq,
                          seqScore(cloneExonSeq, chrExonSeq) );
          /*      printf ( "exon %i %i %s\n", chrExonBeg, chrExonEnd, chrExonSeq);*/
                                                                                                                                                             
        }
        (*numExons)++;
        chrExonSeq[0] = '\0';
        cloneExonSeq[0] = '\0';
        exonSeqSize = 0;
        cloneExonBeg = alignmentScan->clone_pos;
        chrExonBeg = alignmentScan->genome_pos;
                                                                                                                                                             
                                                                                                                                                             
        /* 5' splice junction */
        if ( ( seqSize + 11 ) >= allocatedSeqSize ) {
          /*      printf ("Reallocating from %i to %i seq is %i\n", allocatedSeqSize, allocatedSeqSize+reallocWindow, seqSize);*/
          allocatedSeqSize += maxInt(reallocWindow, 11);
          seqsGrow ( cloneAlignedSeq, chrAlignedSeq, allocatedSeqSize );
        }
        for ( i = 0; i < 3; i++ ) {
          (*chrAlignedSeq)[seqSize] = getLocusBase ( tlocus, prevChr );
          (*cloneAlignedSeq)[seqSize] = '-';
          seqSize++;
          prevChr++;
        }
        for ( i = 0; i < 3; i++ ) {
          (*chrAlignedSeq)[seqSize] = '-';
          (*cloneAlignedSeq)[seqSize] = '-';
          seqSize++;
        }
                                                                                                                                                             
        /* We need to back up the current genome pointer chrPos to the first pos of the 3' splice site*/
        prevChr = alignmentScan->genome_pos - alignmentScan->gap_left - 3;
                                                                                                                                                             
        /* 3' splice junction */
        for ( i = 0; i < 3; i++ ) {
          (*chrAlignedSeq)[seqSize] = getLocusBase ( tlocus, prevChr );
          (*cloneAlignedSeq)[seqSize] = '-';
          seqSize++;
          prevChr++;
        }
                                                                                                                                                             
        /*prevChr = alignmentScan->genome_pos - alignmentScan->gap_left;*/
        /*      printf ("%i\n",  alignmentScan->gap_left);*/
                                                                                                                                                             
        /* all additions after 3' splice junction */
        while ( prevChr <= alignmentScan->genome_pos-1 ) {
          /*    while ( prevChr != alignmentScan->genome_pos-1 && chrIntron[prevChr+1 - *alignBeg] == FALSE ) {*/
          if ( ( seqSize + 3 ) >= allocatedSeqSize ) {
            /*      printf ("Reallocating from %i to %i seq is %i\n", allocatedSeqSize, allocatedSeqSize+reallocWindow, seqSize);*/
            allocatedSeqSize += maxInt(reallocWindow, 3);
            seqsGrow ( cloneAlignedSeq, chrAlignedSeq, allocatedSeqSize );
          }
          if ( ( exonSeqSize + 3 ) >= exonAllocatedSeqSize ) {
            /*      printf ("Reallocating from %i to %i seq is %i\n", allocatedSeqSize, allocatedSeqSize+reallocWindow, seqSize);*/
            exonAllocatedSeqSize += maxInt(reallocWindow, 3);
            seqsGrow ( &cloneExonSeq, &chrExonSeq, exonAllocatedSeqSize );
          }
          (*chrAlignedSeq)[seqSize] = getLocusBase ( tlocus, prevChr );
          (*cloneAlignedSeq)[seqSize] = '-';
          seqSize++;
          chrExonSeq[exonSeqSize] = getLocusBase ( tlocus, prevChr );
          cloneExonSeq[exonSeqSize] = '-';
          exonSeqSize++;
          prevChr++;
        }
      } /* end possibly an intron... */
    }
                                                                                                                                                             
    /* deletions (skips in clone) */
    while ( prevClone != alignmentScan->clone_pos-1 ) {
      if ( ( seqSize + 3 ) >= allocatedSeqSize ) {
        /*      printf ("Reallocating from %i to %i seq is %i\n", allocatedSeqSize, allocatedSeqSize+reallocWindow, seqSize);*/
        allocatedSeqSize += reallocWindow;
        seqsGrow ( cloneAlignedSeq, chrAlignedSeq, allocatedSeqSize );
      }
                                                                                                                                                             
      if ( ( exonSeqSize + 3 ) >= exonAllocatedSeqSize ) {
        exonAllocatedSeqSize += reallocWindow;
        seqsGrow ( &cloneExonSeq, &chrExonSeq, exonAllocatedSeqSize );
      }
                                                                                                                                                             
      prevClone++;
      (*chrAlignedSeq)[seqSize] = deletionChar;
      (*cloneAlignedSeq)[seqSize] = qseq->dna[prevClone];
      seqSize++;
      chrExonSeq[exonSeqSize] = deletionChar;
      cloneExonSeq[exonSeqSize] = qseq->dna[prevClone];
      exonSeqSize++;
    } /* end deletions*/
    /*uuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuuu*/

    /* Finishing with starting polyA tail */
    if (deletionChar == '*')
	deletionChar = '-';

    if ( ( exonSeqSize + 2 ) >= exonAllocatedSeqSize ) {
      exonAllocatedSeqSize += maxInt(reallocWindow, 2);
      seqsGrow ( &cloneExonSeq, &chrExonSeq, exonAllocatedSeqSize );
    }
                                                                                                                                                             
    (*cloneAlignedSeq)[seqSize] = alignmentScan->clone_base;
    (*chrAlignedSeq)[seqSize] = alignmentScan->genome_base;
    seqSize++;
    cloneExonSeq[exonSeqSize] = alignmentScan->clone_base;
    chrExonSeq[exonSeqSize] = alignmentScan->genome_base;
    exonSeqSize++;
                                                                                                                                                             
    /*
    printf("%u\t%c -- %c %u\n",
           alignmentScan->clone_pos,
           alignmentScan->clone_base,
           alignmentScan->genome_base,
           alignmentScan->genome_pos);
    */
    prevChr = alignmentScan->genome_pos;
    prevClone = alignmentScan->clone_pos;
                                                                                                                                                             
    chrExonEnd = alignmentScan->genome_pos;
    cloneExonEnd = alignmentScan->clone_pos;
                                                                                                                                                             
    alignmentScan = alignmentScan->child;
                                                                                                                                                             
    //Check for insertions at the end of the clone
    if ((alignmentScan == 0) && (prevClone < cloneEnd)) {
      if (orientation == '+') {
          if (three_prime_tail_check(qseq, prevClone+1, cloneEnd, orientation))
		deletionChar = '*';
              /* prevClone = cloneEnd; */
      }
      while (prevClone < cloneEnd) {
            if ( ( seqSize + 3 ) >= allocatedSeqSize ) {
        /*      printf ("Reallocating from %i to %i seq is %i\n", allocatedSeqSize, allocatedSeqSize+reallocWindow, seqSize);*/
                allocatedSeqSize += reallocWindow;
                seqsGrow ( cloneAlignedSeq, chrAlignedSeq, allocatedSeqSize );
            }
                                                                                                                                                             
            if ( ( exonSeqSize + 3 ) >= exonAllocatedSeqSize ) {
                exonAllocatedSeqSize += reallocWindow;
                seqsGrow ( &cloneExonSeq, &chrExonSeq, exonAllocatedSeqSize );
            }
                                                                                                                                                             
            prevClone++;
            (*chrAlignedSeq)[seqSize] = deletionChar;
            (*cloneAlignedSeq)[seqSize] = qseq->dna[prevClone];
            seqSize++;
            chrExonSeq[exonSeqSize] = deletionChar;
            cloneExonSeq[exonSeqSize] = qseq->dna[prevClone];
            exonSeqSize++;
      }
    }
  }
                                                                                                                                                             
  /* end alignment scan */
  chrExonSeq[exonSeqSize] = '\0';
  cloneExonSeq[exonSeqSize] = '\0';
                                                                                                                                                             
  /************************************************************************/
  /* create new exon member                                               */
  /************************************************************************/
  /* the next 2 for loops widen the genome coord range of the ends of the exon has any */
        /* deletions in the clone*/
  for ( skipCloneIdx = 0; (cloneExonSeq[skipCloneIdx] == '-' || chrExonSeq[skipCloneIdx] == '-' || chrExonSeq[skipCloneIdx] == '*'); skipCloneIdx++ ) {
        if (cloneExonSeq[skipCloneIdx] == '-')
            chrExonBeg--;
        else
            cloneExonBeg--;
  }
  for ( skipCloneIdx = exonSeqSize-1; (cloneExonSeq[skipCloneIdx] == '-' || chrExonSeq[skipCloneIdx] == '-' || chrExonSeq[skipCloneIdx] == '*'); skipCloneIdx-- ) {
        if (cloneExonSeq[skipCloneIdx] == '-')
            chrExonEnd++;
        else
            cloneExonEnd++;
  }
  if ( orientation == '-' ) {
                                                                                                                                                             
    coordTemp = cloneEnd - cloneExonBeg;
    cloneExonBeg = cloneEnd - cloneExonEnd;
    cloneExonEnd = coordTemp;
    //printf("beg-end : %d-%d\n", cloneExonBeg, cloneExonEnd);                                                                                                                                                        
    reverseComplement ( cloneExonSeq, strlen(cloneExonSeq) );
    reverseComplement ( chrExonSeq, strlen(chrExonSeq) );
    exonAddBegMalloc ( exons, cloneExonBeg, cloneExonEnd,
                       chrExonEnd, chrExonBeg, cloneExonSeq, chrExonSeq,
                       seqScore(cloneExonSeq, chrExonSeq) );
                                                                                                                                                             
  }
  else {
                                                                                                                                                             
    exonAddMalloc ( exons, exons->tail, cloneExonBeg, cloneExonEnd,
                    chrExonBeg, chrExonEnd, cloneExonSeq, chrExonSeq,
                    seqScore(cloneExonSeq, chrExonSeq) );
    /*    printf ( "exon %i %i %s\n", chrExonBeg, chrExonEnd, chrExonSeq);*/
  }
                                                                                                                                                             
  (*numExons)++;
  chrExonSeq[0] = '\0';
  cloneExonSeq[0] = '\0';
  exonSeqSize = 0;
                                                                                                                                                             
  (*cloneAlignedSeq)[seqSize] = '\0';
  (*chrAlignedSeq)[seqSize] = '\0';
                                                                                                                                                             
  if ( orientation == '-' ) {
    reverseComplement( *cloneAlignedSeq, strlen(*cloneAlignedSeq) );
    reverseComplement( *chrAlignedSeq, strlen(*chrAlignedSeq) );
  }
  free(chrExonSeq);
  free(cloneExonSeq);
  /************************************************************************/
  return retval;
}  
/* end collect info */

static int seqGrow ( char **seq, int newSeqSize ) {
  if ( ( (*seq) = (char *)realloc((*seq), newSeqSize) ) == NULL ) {
    printf ("ERROR No space to realloc!\n");
    return 1;
  }
  return 0;
}

static int seqsGrow ( char **seqOne, char **seqTwo, int newSeqSize ) {
  if ( seqGrow ( seqOne, newSeqSize ) == 1 ) {
    printf ("Error!  Not enough space for seqOne!\n");
    return 1;
  }
  if ( seqGrow ( seqTwo, newSeqSize ) == 1 ) {
    printf ("Error!  Not enough space for seqTwo!\n");
    return 1;
  }
  return 0;
}

int maxInt ( int a, int b ) {
  return ( a > b ) ? a : b;
}

static int seqScore ( char *seqA, char *seqB ) {
  int i;
  double matches = 0;
  int len = strlen(seqA);
  if (  strlen(seqB) != len ) {
    printf ("ERROR - String lengths do not match!\n");
  }
  for ( i = 0; i < len; i++ ) {
    if ( seqA[i] == seqB[i] )
      matches++;
  }
  return matches/i * 100;
}
