/*                        PhyloGibbs                                  */

/*   Algorithm developed by Rahul Siddharthan, Erik van Nimwegen      * 
 *   and Eric D. Siggia at The Rockefeller University, New York       * 
 *                                                                    *
 *   This code copyright (C) 2004 Rahul Siddharthan <rsidd@online.fr> * 
 *   Licensed under the GNU General Public License (see COPYING)      */ 

/* 
 * $Author: rsidd $  
 * $Date: 2005/05/02 08:54:09 $ 
 * $Id: gibbsshiftmove.c,v 1.1 2005/05/02 08:54:09 rsidd Exp $ 
 */

#include "interspecies.h"
#include "moveshiftweight.h"
#include "nonselfblocked.h"
#include "fullbinweight.h"
#include <math.h>
#include <gsl/gsl_rng.h>
#include "binbasecount.h"

/*  This samples states obtained by shifting all windows in one colour
 *  by one unit, left or right.  If any of the to-be-shifted-to windows are 
 *  blocked, or their length definition differs, the proposed move is 
 *  abandoned.
 */

int gibbsshiftmove(params *v, double *logprevweight) 
{
    int ncol,nbin,n,m,nc,nb,nm,ns,nss,oldcol,zer,rejectthismove,one;
    oneshiftmove trialmove, *thismove; 
    GArray *moves;
    window *currwin,*currwin2,*currwin3;
    double totalweight,moverandom,currweight,truefullbinweight,maxweight;
    GPtrArray *binwin0,*binwin1,*currbin;

    moves=g_array_new(TRUE,TRUE,sizeof(oneshiftmove));

    maxweight = -10000.0;
    zer=0;
    one=1;

    if ((v->bin)->len==1)  /* no windows selected, return*/
        return 1;


    /**pick a color at random****/
    ncol=ncolours(v->bin);
    nc = 1+gsl_rng_uniform_int(v->gslrand, ncol-1);
    nbin=bincount(v->bin,nc);
    /**keep going until nonzero color***/
    while(nbin == 0)
      {
	 nc = 1+gsl_rng_uniform_int(v->gslrand, ncol-1);
	 nbin=bincount(v->bin,nc);
      }
    trialmove.bin=nc;
    currbin=binwinset((v->bin),nc);

    rejectthismove=0;
    ns=0;
    
    /* left shift */
    
    while (!rejectthismove) {
      ns++;
      trialmove.newbin=g_ptr_array_new();
      one=1;
    
      /* check if the window collides with any windows
       * to the left or right */
      rejectthismove=0;
      for (nb=0; (nb<nbin)&&(!rejectthismove); nb++) {
	currwin=binwindow(currbin,nb);
	
	if (currwin->dir) { /* reverse complement */
	  nss=0;
	  currwin2=currwin;
	  do {
	    currwin2=g_ptr_array_index(currwin2->right,0);
	    if ((currwin2==NULL)||(currwin2->colour))
	      rejectthismove=1;
	    nss++;
	  } while ((nss<ns)&&(!rejectthismove));
	  if ((currwin2!=NULL)&&(ns<v->wwidth/2))
	    assert(currwin2==g_ptr_array_index(currwin->right,ns-1));
	} else { /* not reverse comp */
	  nss=0;
	  currwin2=currwin;
	  do {
	    currwin2=g_ptr_array_index(currwin2->left,0);
	    if (currwin2==NULL)
	      rejectthismove=1;
	    else if (currwin2->colour) 
	      rejectthismove=1;
	    nss++;
	  } while ((nss<ns)&&(!rejectthismove));
	  if ((currwin2!=NULL)&&(ns<v->wwidth/2))
	    assert(currwin2==g_ptr_array_index(currwin->left,ns-1));
	} 
	
	if (!rejectthismove){
	  /**reject if blocked****/
	  if (nonselfblocked(currwin2,nc)||currwin2->permblocked
	      ||(currwin2->blockers->len>1)||(currwin2->shiftblocked))
	    rejectthismove=1;
	  /* check that it has the same number of sequences
	   * in alignment */
	  else if (currwin2->start->len!=currwin->start->len)
	    rejectthismove=1;
	  else {
	    /* one blocker allowed */
	    assert(currwin2->blockers->len <= 1);
	    g_ptr_array_add(trialmove.newbin,currwin2);
	    for (n=0; n<currwin2->blockedwins->len; n++) {
	      currwin3=g_ptr_array_index(currwin2->blockedwins,n);
	      currwin3->shiftblocked=1;
	    }
	  }
	}
      }
      
      if (!rejectthismove) {
	
	trialmove.moveweight=moveshiftweight(v,&trialmove);
	if (trialmove.moveweight > maxweight)
	  maxweight = trialmove.moveweight;
	binwin1=g_ptr_array_index((v->bin),trialmove.bin);
	assert(trialmove.newbin->len==currbin->len);
	if (trialmove.newbin->len > 0) {
	  for (n=0; n<trialmove.newbin->len; n++) {
	    currwin=g_ptr_array_index(trialmove.newbin,n);
	    for (m=0; m<currwin->blockedwins->len; m++) {
	      currwin2=g_ptr_array_index(currwin->blockedwins,m);
	      currwin2->shiftblocked=0;
	    }
                   }
	  g_array_append_val(moves,trialmove);
	}
      } else {
	for (n=0; n<trialmove.newbin->len; n++) {
	  currwin=g_ptr_array_index(trialmove.newbin,n);
	  for (m=0; m<currwin->blockedwins->len; m++) {
	    currwin2=g_ptr_array_index(currwin->blockedwins,m);
	    currwin2->shiftblocked=0;
	  }
	}
	g_ptr_array_free(trialmove.newbin,TRUE);
      }
      
    }
    
    /* right shift */
    rejectthismove=0;
    ns=0;
    
    while (!rejectthismove) {
      ns++;
      
      trialmove.newbin=g_ptr_array_new();
      rejectthismove=0;
         
      for (nb=0; (nb<nbin)&&(!rejectthismove); nb++) {
	currwin=binwindow(currbin,nb);
	
	if (currwin->dir) { /* reverse complement */
	  nss=0;
	  currwin2=currwin;
	  do {
	    currwin2=g_ptr_array_index(currwin2->left,0);
	    if ((currwin2==NULL) || (currwin2->colour))
	      rejectthismove=1;
	    nss++;
	  } while ((nss<ns)&&(!rejectthismove));
	  if ((currwin2!=NULL)&&(ns<v->wwidth/2))
	    assert(currwin2==g_ptr_array_index(currwin->left,ns-1));
	} else { /* not reverse comp */
	  nss=0;
	  currwin2=currwin;
	  do {
	    currwin2=g_ptr_array_index(currwin2->right,0);
	    if ((currwin2==NULL) || (currwin2->colour))
	      rejectthismove=1;
	    nss++;
	  } while ((nss<ns)&&(!rejectthismove));
	  if ((currwin2!=NULL)&&(ns<v->wwidth/2))
	    assert(currwin2==g_ptr_array_index(currwin->right,ns-1));
	}
	
	if (!rejectthismove){
	  if (nonselfblocked(currwin2,nc)||currwin2->permblocked
	      ||(currwin2->blockers->len>1)||(currwin2->shiftblocked))
	    rejectthismove=1;
	  else if (currwin2->start->len!=currwin->start->len)
	    rejectthismove=1;
	  else {
	    assert(currwin2->blockers->len <= 1);
	    g_ptr_array_add(trialmove.newbin,currwin2);
	    for (n=0; n<currwin2->blockedwins->len; n++) {
	      currwin3=g_ptr_array_index(currwin2->blockedwins,n);
	      currwin3->shiftblocked=1;
	    }
	  }
	}
      }
      
      
      if (!rejectthismove) {
	binwin1=g_ptr_array_index((v->bin),trialmove.bin);
	trialmove.moveweight=moveshiftweight(v,&trialmove);
	if (trialmove.moveweight > maxweight)
	  maxweight = trialmove.moveweight;
	assert(trialmove.newbin->len==currbin->len);
	if (trialmove.newbin->len > 0) {
	  for (n=0; n<trialmove.newbin->len; n++) {
	    currwin=g_ptr_array_index(trialmove.newbin,n);
	    for (m=0; m<currwin->blockedwins->len; m++) {
	      currwin3=g_ptr_array_index(currwin->blockedwins,m);
	      currwin3->shiftblocked=0;
	    }
	  }
	  g_array_append_val(moves,trialmove);
	}
      } else {
	for (n=0; n<trialmove.newbin->len; n++) {
	  currwin=g_ptr_array_index(trialmove.newbin,n);
	  for (m=0; m<currwin->blockedwins->len; m++) {
	    currwin2=g_ptr_array_index(currwin->blockedwins,m);
	    currwin2->shiftblocked=0;
	  }
	}
	g_ptr_array_free(trialmove.newbin,TRUE);
      }
    }

    /* null move -- we may not want to do a shift at all */
    
    trialmove.bin=-1;
    trialmove.newbin=g_ptr_array_new();
    trialmove.moveweight=0.0;
    if (trialmove.moveweight > maxweight)
      maxweight = trialmove.moveweight;
    g_array_append_val(moves,trialmove);
    
    /* normalise the weights */
    totalweight = 0.0;
    for (nm=0; nm<(moves->len); ++nm) {
      thismove = &g_array_index(moves,oneshiftmove,nm);
      thismove->moveweight = exp((v->beta)*(thismove->moveweight-maxweight));
      totalweight += thismove->moveweight;
    }
    
    
    /* now pick a move from moves, weighted by moveweight */
    
    moverandom=gsl_rng_uniform(v->gslrand)*totalweight;
    
    currweight=0.0;
    nm=0;
    do {
      trialmove=g_array_index(moves,oneshiftmove,nm);
      nm++;
      currweight=currweight+trialmove.moveweight;
    } while (currweight<moverandom);
    
    if (trialmove.bin==-1)
      return 0;
    
    *logprevweight=*logprevweight+log(trialmove.moveweight)
      + (v->beta)*maxweight;
    
    binwin1=g_ptr_array_index((v->bin),(trialmove.bin));
    assert(trialmove.bin!=0);
    
       
    binwin0=g_ptr_array_index((v->bin),zer);
    assert((binwin1->len) >0);
    for (n=binwin1->len-1; n>=0; n--) {
      currwin=g_ptr_array_index(binwin1,n);
      currwin2=g_ptr_array_index(trialmove.newbin,n);
      currwin2->dir=currwin->dir;
      currwin->colour=0;
      assert(!currwin->blocked);
      g_ptr_array_add(binwin0,currwin);
      for (nb=0; nb<currwin->blockedwins->len; nb++) {
	currwin2=g_ptr_array_index(currwin->blockedwins,nb);
	g_ptr_array_remove_fast((currwin2->blockers),currwin);
	if (currwin2->blockers->len==0)
	  currwin2->blocked=0;
	if (currwin2->colour==trialmove.bin)
	  assert(!currwin2->blocked);
      }
      g_ptr_array_remove_index(binwin1,n);
    }
    assert((binwin1->len)==0);
    
    for (n=0; n<trialmove.newbin->len; n++) {
      currwin=g_ptr_array_index(trialmove.newbin,n);
      oldcol=currwin->colour;
      assert(oldcol==0);
      binwin0=g_ptr_array_index((v->bin),oldcol);
      g_ptr_array_remove_fast(binwin0,currwin);
#ifdef DEBUG
      for (nc=0; nc<binwin0->len; nc++)
	assert(((window *)((binwin0->pdata)[nc]))!=currwin);
      for (nc=0; nc<binwin1->len; nc++)
	assert(((window *)((binwin1->pdata)[nc]))!=currwin);
#endif
      g_ptr_array_add(binwin1,currwin);
      currwin->colour=trialmove.bin;
      assert(!currwin->blocked);
      for (nb=0; nb<currwin->blockedwins->len; nb++) {
	currwin2=g_ptr_array_index(currwin->blockedwins,nb);
	assert(currwin2->colour==0);
	g_ptr_array_add((currwin2->blockers),currwin);
	currwin2->blocked=1;
	assert(currwin2->blockers->len>0);
      }
    }
    truefullbinweight=fullbinweight(v);
    fflush(NULL);
    if(fabs((*logprevweight)-truefullbinweight)  > 0.1)
      fprintf(stderr,"WARNING: abnormally large difference between the calculated score %lf and\n the true full score %lf in gibbsshiftmove\n",*logprevweight,truefullbinweight);
    *logprevweight = truefullbinweight;
    
    binwin1=g_ptr_array_index((v->bin),trialmove.bin);
    assert((binwin1->len)==trialmove.newbin->len);
    
    for (n=0; n<moves->len; n++) {
      trialmove=g_array_index(moves,oneshiftmove,n);
      g_ptr_array_free(trialmove.newbin,TRUE);
    }
    g_array_free(moves,TRUE); 
    
    makebinbasecount(v);
    
    return 0;
    
}    
