/*                        PhyloGibbs                                  */

/*   Algorithm developed by Rahul Siddharthan, Erik van Nimwegen      * 
 *   and Eric D. Siggia at The Rockefeller University, New York       * 
 *                                                                    *
 *   This code copyright (C) 2004 Rahul Siddharthan <rsidd@online.fr> * 
 *   Licensed under the GNU General Public License (see COPYING)      */ 

/* 
 * $Author: rsidd $  
 * $Date: 2005/05/22 12:45:08 $ 
 * $Id: main.c,v 1.5 2005/05/22 12:45:08 rsidd Exp $ 
 */

/*** This edit begun on Oct 13, 2004 by Erik
     tasks: 
     1) change anneal strategy to have a single anneal with a fixed length of time + param annealiters
     2) add a warm up phase at beta=1.0 + param: transientiters
     3) make beta increase additively.
****/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include <sys/time.h>
#include <glib.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_sf_gamma.h>

#include "interspecies.h"

#include "binbasecount.h"
#include "binprint.h"
#include "binscore.h"
#include "commonroutines.h"
#include "fasta.h"
#include "fullbinweight.h"
#include "gibbscolourmove.h"
#include "gibbsshiftmove.h"
#include "switchwindow.h"
#include "initroutines.h"
#include "parseopts.h"
#include "printhelp.h"
#include "readDialign.h"
#include "readfasta.h"
#include "recolourwindow.h"
#include "seq2nums.h"
#include "setbgcount.h"
#include "setwinbasecount.h"
#include "trackedbinprint.h"
#include "updatetracked.h"
#include "windowscore.h"
#include "wminteg.h"
#include "initbinnumconfig.h"

/* macro to do a move a specified number of times */
#define do_n_moves(ntimes,movevar,doavg,movefunc) do {for (n=1; n<=ntimes; n++) { movefunc; v.totmoves++; movevar++;} } while (0)

int main(int argc, char *argv[]) {
    
    int n1,n,m,zer,one,k,oldnwin,oldnbin,nprog,jjj,extraregion;
    /*int newnwin, newnbin;*/
    long int i,stop;
    double half,fzer, logprevweight;
    params v;
    GPtrArray *binwin1,*binwin2;
    GArray *bindir;
    window *currwin;
      
    zer=0;
    one=1;
    fzer = 0.0;
    half=0.5;
    jjj = 0;
    /* recover command line */
    strncpy(v.arguments,"",1);
    for (n=1; n<argc; n++) {
        strncat(v.arguments,argv[n],1023-strlen(v.arguments));
        strncat(v.arguments," ",1023-strlen(v.arguments));
    }
    
    /* set default values for everything */
    initvalues(&v);
    
    /* read command line options, overrule both defaults and env vars */
    if (parseopts(argc,argv,&v))
        return 1;
    
    if (v.gethelp) {
        printhelp();
        return 0;
    }
    
    /* post-process various parameters */
    if(!v.quiet)
      printf("processing input values\n");
    if (processvalues(&v))
        return 1;


    
    /**tables with logs of gamma ratios**/
    if(!v.quiet)
      printf("initializing log-gamma tables\n");
    inittables(v.pseudocount);
    
    /* initialise the windows structure */
    if(!v.quiet)
      printf("setting up windows\n");
    if (setupwindows(&v))
        return 1;

    /***if number of groups=1, extraregion=0 else extraregion=1****/
    if((v.groupstarts)->len <= 2)
      extraregion=0;
    else
      extraregion=1;

    logprevweight=fullbinweight(&v);
    
    if (strlen(v.posfile)>0)
        binprint(&v,"initial_output");
    
    /**automatic setting of betaincr******/
    /**we use the following formula to determine the value of beta_final:**/
    /**beta_final = log[10 N]/(w log[8/5]) *****/
    /**where  w is the width of the motif, N the total number of windows and 
       8/5 is the ratio of expected score site to own WM and expected score random sequence to WM**/
    if(v.betaincr < -0.00001){
        v.betaincr = log((double) (10 * (v.win->len -1) *((v.bin)->len-1)));
        v.betaincr = v.betaincr/((double) (v.wwidth));
        v.betaincr = v.betaincr/0.470003629;
        /**now divide by the total number of steps**/
        v.betaincr = v.betaincr/((double) (v.annealiters));
    }
    
    /*prints parametr settings to standard out**/
    if (v.verbose)
        initprint(&v);
    
    v.totmoves=0;
    v.tcsteps=0;
    v.twsteps=0;
    v.tssteps=0;
    v.tmsteps=0;

    /* With colour moves, number of coloured windows can change
       from when initial guess for Ncolm Nwinm etc were made.  To
       account for this, scale iters by newnwin/oldnwin */
    oldnwin=0;
    for (n=1; n<v.bin->len; n++) {
        binwin1=g_ptr_array_index(v.bin,n);
        oldnwin += binwin1->len;
    }
    oldnbin=(v.bin)->len - 1;
    
    if(!v.quiet)
      printf("Cycling: %d color moves, %d window moves, %d shift moves\n",v.Ncolm,v.Nwinm,v.Nshiftm);

    /***there are 3 phases: transient, anneal, and deep quench ***/
    for(k=0;k<3;++k){
        if (k==0) {
            stop = v.transientiters;
            if (!v.quiet)
                printf("\nStarting transient equilibriation phase\n");
        }
        else if(k ==1){
            stop = v.annealiters;
            if (v.trackedbins != NULL)
                stop=0; /* No anneal if tracking specified bins */
            if (stop>0) {
                if (!v.quiet)
                    printf("\nStarting simulated anneal phase\n");
            }
        }
        else{
            stop = v.deepquenchiters;
            if (v.trackedbins != NULL)
                stop = 0; /* No quench if tracking specified bins */
            /**set beta=20 for deep quench**/
            logprevweight /= v.beta;
            v.beta = 20.0;
            logprevweight *= v.beta;
            if (stop > 0) {
                if (!v.quiet)
                    printf("\nStarting deep-quench phase\n");
            }
        }
        if ((v.verbose)&&(stop>0))
            printf("Doing %ld iterations of %d colour-change, %d window-shift, %d global-shift\n",stop,v.Ncolm,v.Nwinm,v.Nshiftm);

        if ((!v.quiet)&&(stop>0)) {
            show_progress_bar(0,stop);
        }
        i=0;
        while(i< stop){
            if(k==1){
                /**during anneal increase beta linearly**/
                logprevweight /= v.beta;
                v.beta += v.betaincr;
                logprevweight *= v.beta;
            }
	   
	    do_n_moves(v.Ncolm,v.tcsteps,0,gibbscolourmove(&v,&logprevweight));
                        
	    do_n_moves(v.Nwinm,v.twsteps,1,switchwindow(&v,&logprevweight,extraregion));
            
            do_n_moves(v.Nshiftm,v.tssteps,0,gibbsshiftmove(&v,&logprevweight));
                    
	    ++i;
            if (!(v.quiet)) {
                show_progress_bar(i,stop);
            }
            if (!v.writeatend)
                binprint(&v,"");
        }
	/**remove this for now. Shouldn't change anyway if prior is working***/
        /*if ((k==0) && (v.Ncolm != 0)) {*/
            /* adjust total number of moves by the increase in
             * number of coloured windows or colours*/
            /*newnwin=0;*/
            /*for (n=1; n<v.bin->len; n++) {*/
	/*binwin1=g_ptr_array_index(v.bin,n);*/
	/*newnwin += binwin1->len;*/
	/*}*/
	/*newnbin= (v.bin->len) - 1;*/
	/*v.Nwinm = (v.Nwinm * newnwin)/oldnwin;*/
	/*v.Nshiftm = (v.Nshiftm * newnbin)/oldnbin;*/
        /*}*/
        if ((!v.quiet)&&(stop>0))
            printf("| Done.\n");
    }
    
    /***write the anneal results****/
    binprint(&v,"");
    
    
    /***now the tracking phase (if requested) *****/
    if(v.autotrack){
        /***set up the tracked bins*****/
        v.betaincr=0.0;
        v.beta = 1.0;
	/**average log-prob over all time steps***/
	v.countent = 0;
	v.meanent = 0.0;
        
        if (v.trackedbins!=NULL) {
            v.transientiters = 0; /* We've already done transient iters and no
                                   * anneal/quench in this case */
            if (!v.quiet)
                printf("\nTracking specified window clusters\n");
            if (v.verbose)
                printf("Doing %ld iterations of %d colour-change, %d window-shift, %d global-shift,\n",v.totaliters,v.Ncolm,v.Nwinm,v.Nshiftm);
        }
        else 
	  {
	    if (v.bin->len ==1) 
	      {
		fprintf(stderr,"\nConfiguration at the end of anneal has zero windows!\nSkipping tracking phase: nothing found\n");
		return 1;
	      }
            v.trackedbins=g_ptr_array_new();
	    v.trackeddirs=g_ptr_array_new();
	    /**note: tracked bin 0 corresponds to color with best scoring window***/
            for (n1=0; n1<v.binnums->len; n1++) {
	      n=g_array_index(v.binnums,int,n1);
	      binwin1=g_ptr_array_index(v.bin,n);
	      binwin2=g_ptr_array_new();
	      bindir = g_array_new(FALSE,FALSE,sizeof(int));
	      for (m=0; m<binwin1->len; m++) {
		currwin=g_ptr_array_index(binwin1,m);
		g_ptr_array_add(binwin2,currwin);
		g_array_append_val(bindir,currwin->dir);
	      }
	      g_ptr_array_add(v.trackedbins,binwin2);
	      g_ptr_array_add(v.trackeddirs,bindir);
            }
            if (!v.quiet)
	      printf("\nTracking annealed window clusters\n");
            if (v.verbose)
	      printf("Doing %ld transient and %ld final iterations of %d colour-change,\n  %d window-shift, %d global-shift\n",v.transientiters,v.totaliters,v.Ncolm,v.Nwinm,v.Nshiftm);
            /***randomize the current state****/
            if (v.occset != NULL){
	      /**random state***/
	      if(initbinnumconfig(&v))
		{
		  fprintf(stderr,"Randomization at start of tracking failed, exiting\n");
		  return 1;
		}
	    }
	  }
        
        
        /**recalculate the score***/
        v.beta = 1;
        logprevweight = fullbinweight(&v);
        
        /***do the moves, including a transient phase*******/
        i = 0;
        if (!v.quiet) {
            show_progress_bar(0,(v.transientiters+v.totaliters));
        }
        while(i < (v.totaliters+v.transientiters)){
	  /**update score average***/
	  v.meanent += logprevweight;
	  ++v.countent;

	  do_n_moves(v.Ncolm,v.tcsteps,0,gibbscolourmove(&v,&logprevweight));
	  
	  do_n_moves(v.Nwinm,v.twsteps,1,switchwindow(&v,&logprevweight,extraregion));
	  
	  do_n_moves(v.Nshiftm,v.tssteps,0,gibbsshiftmove(&v,&logprevweight));
	 	  
	  ++i;
	  if (!v.quiet) {
	    show_progress_bar(i,(v.transientiters+v.totaliters));
	  }
	  /**update and output tracking results***/
	  if (i> v.transientiters) {
	    ++jjj;
	    updatetracked(&v);
	  }
	  if (!v.writeatend) /**tracking output**/
	    trackedbinprint(&v,i-v.transientiters);
        }
        /**tracking output**/
        trackedbinprint(&v,i-v.transientiters);
    }
    if (!v.quiet)
        printf("\nDone.\n");

    return (0);
}
