/*****************************************************************
        Copyright by Rockefeller University,
can not be reproduced or distributed without written permission of
copyright holder.  Version of October 2003.

Written by Saurabh Sinha (contact person), Erik van Nimwegen, and 
Eric Siggia.

The program stubb (and its relatives) implement an algorithm for
finding likely cis-regulatory modules, described in the following
paper:
"A Probabilistic Method to Detect Regulatory Modules"
by Saurabh Sinha, Erik van Nimwegen and Eric Siggia. 
Eleventh International Conference on Intelligent Systems for
Molecular Biology, Brisbane, Australia, July 2003, pg 292-301.

The file sample/gap_wtmx that comes with this distribution includes 
a sample set of transcription factor weight matrices (PWM's) that 
were reported in :
"Computational detection of genomic cis-regulatory modules applied
to body patterning in the early Drosophila embryo"
by N. Rajewsky, M. Vergassola, U. Gaul and E. Siggia.
BMC Bioinformatics 3 (30) 2002.
******************************************************************/

#include "parameters.h"
#include <math.h>
#include <stdexcept>
#include "util.h"

void Parameters_H1::Train(bool differential)
{
  if (!_is_initialized) {
    printf("Training attempted without initialization\n");
    _is_trained = false;
    return;
  }

  DTYPE freeEnergyBackground =  EvaluateFreeEnergyBackground();

  DTYPE  previousFreeEnergy = INF_FREE_ENERGY;
  _free_energy = EvaluateFreeEnergy();

  int count = 0;
  do {
    count++;
    previousFreeEnergy = _free_energy;
    Update();
    _free_energy = EvaluateFreeEnergy();  
    _free_energy_differential = freeEnergyBackground - _free_energy;
    if (count > CHECK_ITERATION_THRESHOLD && _free_energy_differential < CHECK_FEN_THRESHOLD) break;    
    if (count > MAX_TRAINING_ITERATIONS) break;    
  } while (previousFreeEnergy - _free_energy > THRESHOLD);

  _num_iterations = count;        // record how many iterations were needed

  if (differential) {
    _free_energy_differential = freeEnergyBackground - _free_energy;
    if (_free_energy_differential < 0) _free_energy_differential = 0;
  }
  else _free_energy_differential = 0;

  int numSpecific = 0;
  int numTotal = 0;
  for (int i=0; i<_windows->size(); i++) {
    Window *win = (*_windows)[i];
    numSpecific += win->NumSpecificCharacters();
    numTotal += win->Length();
  }

#ifdef _NORMALIZE_SPACERS
  if (numSpecific == 0) _free_energy_differential = 0;
  else _free_energy_differential *= float(numTotal)/float(numSpecific);
#endif

  _free_energy_perlength = _free_energy_differential / float(numTotal);

  _is_trained = true;
}

void Parameters_H1::Initialize(vector<Window *> *wl, WtMxCollection *wmc, int initialbias)
{
  _initialbias = initialbias;
  if (initialbias < 0) 
    Initialize(wl,wmc,(Parameters *)NULL);
  else {
    Parameters_H1 seed(wmc, initialbias);
    Initialize(wl,wmc,&seed);
  }                          
}

void Parameters_H1::UpgradeInitialize(vector<Window *> *wl, WtMxCollection *wmc, Parameters_H0 *init)
{
  Parameters_H1 seed(wmc,0);  // 0 is for bias towards first motif. Doesnt matter, since it'll be reset
  int numWM = init->NumWM();
  for (int i=0; i<numWM; i++) {
    for (int j=0; j<numWM; j++) {
      seed._pij[i][j] = init->_pi[j];
    }
  }
  Initialize(wl,wmc,&seed);
  WtMx *bkgwm;
  init->GetBackground(bkgwm);
  SetBackground(new WtMx(bkgwm));
  CacheBackgroundProbabilities();
}

#ifdef _CYCLIC_WINDOWS
void Parameters_H1::SetInitial(float **initial)
{
  int numWindows = _windows->size();
  int numWM = _numWM;

  // first delete if already present
  if (_initial) {
    for (int wi=0; wi<numWindows; wi++) 
      delete [] _initial[wi];
    delete [] _initial;
  }

  if (initial == NULL) {
    _initial = NULL;
    return;
  }

  // then allocate space and copy
  _initial = new float *[numWindows];
  for (int wi=0; wi<numWindows; wi++) {
    _initial[wi] = new float[numWM];
    for (int j=0; j<numWM; j++) {
      _initial[wi][j] = initial[wi][j];
    }
  }

  return;
}
#endif

void Parameters_H1::DoDynamicProgramming()
{
  // Compute the \alphas, \betas, \Aijl from these parameter values, dont change
  // the parameter values
  int numWindows = _windows->size();
  for (int wi=0; wi<numWindows; wi++) {
    Window *window = _currentWindow = (*_windows)[wi];   
    Forward();          // alpha
    Backward();         // beta
    PrepareForUpdate(); // change Aijl, but dont update pij !

    DTYPE  term = FringeCorrectionFactorHelper(window);
    _fringe_corrections[wi] = 1/term;

  }
}

void Parameters_H1::Initialize(vector<Window *> *wl, WtMxCollection *wmc, Parameters *initseed)
{
  int i;

  Destroy();                      // destroy whatever initialization that may already have occurred

  _windows = wl;
  _wmc = wmc;
  _numWM = _wmc->Size()+1;
  _wm_len = new int[_numWM];

  Parameters_H1 *init = (Parameters_H1 *)initseed;
  int bkgIndex =  BackgroundIndex(_wmc);     
  TrainBackground();

                                              // create the record for which weight matrices are free
  _free_emission_probabilities = new bool *[_numWM];
  int max_wm_len = -1;
  for (i=0; i<_numWM; i++) {
    WtMx *w;
    if (i==bkgIndex) 
      w = _bkgwm;
    else
      w = _wmc->WM(i);

    int wm_len = w->Length();
    _wm_len[i] = wm_len;
    if (wm_len > max_wm_len) max_wm_len = wm_len;

    _free_emission_probabilities[i] = new bool[wm_len];
    for (int j=0; j<w->Length(); j++) {
      _free_emission_probabilities[i][j] = false;
    }
  }
  if (max_wm_len < 0) {
    printf("Error: max_wm_length is < 0\n");
    exit(1);
  }

  _pij = new float *[_numWM];
  _oldpij = new float *[_numWM];
  for (i=0; i<_numWM; i++) {
    _pij[i] = new float[_numWM];
    _oldpij[i] = new float[_numWM];
    for (int j=0; j<_numWM; j++) {
      if (init==NULL) {
	_pij[i][j] = 1/float(_numWM); // initialize to the uniform distribution
      }
      else {                          // initialize to the parameters in init 
	_pij[i][j] = init->_pij[i][j];
      }
      _oldpij[i][j] = _pij[i][j]; 
    }
  }

  _max_window_length = -1;
  int numWindows = wl->size();
  _Aij = new DTYPE  **[numWindows];
  _fringe_corrections = new DTYPE [numWindows];
  for (int wi=0; wi<numWindows; wi++) {
    Window *window = (*wl)[wi];
    int length = window->Length();
    
    _Aij[wi] = new DTYPE  *[_numWM];
    for (i=0; i<_numWM; i++) {
      _Aij[wi][i] = new DTYPE [_numWM];
      for (int j=0; j<_numWM; j++) {
	_Aij[wi][i][j] = 0; 
      }
    }
      
    if (_max_window_length < length) _max_window_length = length;
    _fringe_corrections[wi] = -1.0;
  }
  if (_max_window_length < 0) {
    printf("Error: maximum window length < 0\n");
    exit(1);
  }

  _Aijl_window = new DTYPE **[_numWM];
  for (i=0; i<_numWM; i++) {
    _Aijl_window[i] = new DTYPE *[_numWM];
    for (int j=0; j<_numWM; j++) {
      _Aijl_window[i][j] = new DTYPE[_max_window_length];
    }
  }

  _alpha = new DTYPE   *[_max_window_length];
  _beta  = new DTYPE   *[_max_window_length];
  _c     = new DTYPE    [_max_window_length];
  _cij  = new DTYPE    *[_max_window_length];
  for (i=0; i<_max_window_length; i++) {
    _alpha[i] = new DTYPE [_numWM];
    _beta[i]  = new DTYPE [_numWM];
    _c[i] = 1;
    _cij[i] = new DTYPE [max_wm_len];
  }

#ifdef _CYCLIC_WINDOWS
  _initial = NULL;
#endif

  _is_initialized = true;
}

DTYPE  Parameters_H1::EvaluateFreeEnergy()
{
  if (!_is_initialized) {
    printf("Error: EvaluateFreeEnergy attempted without complete initialization\n");
    exit(1);
  }

  DTYPE  logP = 0;
  int numWindows = _windows->size();
  for (int wi=0; wi<numWindows; wi++) {
    Window *window = _currentWindow = (*_windows)[wi];
    Forward();                         // update the _alpha's and the c's
    Backward();                        // update the _beta's
    PrepareForUpdate();                // update the Aijl's
                                       
                                       // now evaluate the logP
    int length = window->Length();
    DTYPE  tmplogP = 0;
    for (int i=0; i<length; i++) {
      tmplogP += (-(log(_c[i])));
    }
                                       // also add the fringe occurrences
    DTYPE  term = FringeCorrectionFactorHelper(window);
    _fringe_corrections[wi] = 1/term;
    tmplogP += log(term);              // tmplogP is the contribution from the current window
    logP += tmplogP;
  }

  return -logP;    
}

DTYPE  Parameters_H1::FringeCorrectionFactorHelper(Window *window)
{
  int bkgIndex = BackgroundIndex();
  WtMx **wm = new WtMx *[_numWM];
  for (int i=0; i<_numWM; i++) {
    if (i!=bkgIndex) {
      wm[i] = _wmc->WM(i);
    }
    else {
      wm[i] = _bkgwm;
    }
  }
  
  int length = window->Length();
  DTYPE  term = 1.;
  for (int j=0; j<_numWM; j++) {
    int wm_len_j = _wm_len[j];
    if (j==bkgIndex || wm_len_j==1) continue;
    for (int k=1; k<wm_len_j; k++) {
      if (window->AmbiguousCharAt(length-1-k+1)) continue;
      for (int i=0; i<_numWM; i++) {
	DTYPE  scale_factor = _cij[length-1-k+1][k-1];
	DTYPE  factor = _alpha[length-1-k][i]*scale_factor*_pij[i][j]*ComputeSequenceProbability(window,length-1-k+1,length-1,wm[j]);
	term += factor;
      }
    }
  }
                                    // Clean up
  delete [] wm;

  return term;
}

DTYPE  Parameters_H1::FringeCorrectionFactor(int index)
{
  if (!_is_initialized) {
    printf("Error: FringeCorrectionFactor attempted without complete initialization\n");
    exit(1);
  }

  int numWindows = _windows->size();
  if (index >= numWindows) {
    printf("Error: FringeCorrectionFactor attempted on non-existing window\n");
    exit(1);
  }

  if (_fringe_corrections[index] > -0.5) {   // before being computed, this is at -1.
    return _fringe_corrections[index];
  }
  else {
    printf("Error: FringeCorrectionFactor doesnt have required information\n");
    exit(1);
  }
}

void Parameters_H1::Forward()
{
  int i;
  int bkgIndex = BackgroundIndex(); // the background matrix corresponds to last index
  WtMx **wm = new WtMx *[_numWM];
  int max_wm_length = 0;
  for (i=0; i<_numWM; i++) {
    if (i!=bkgIndex) {
      wm[i] = _wmc->WM(i);
    }
    else {
      wm[i] = _bkgwm;
    }
    if (_wm_len[i] > max_wm_length) max_wm_length = _wm_len[i];
  }

  Window *window = _currentWindow;
  int length = window->Length();
#ifdef _CYCLIC_WINDOWS
  int windowindex = window->GetIndex();
  if (_initial == NULL) {
    printf("Error: _initial null for cyclic windows\n");
    exit(1);
  }
#endif
                                    // Clear up previous information
  for (int l=0; l<length; l++) {
    for (i=0; i<_numWM; i++) {
      _alpha[l][i] = 0;
      }
      _c[l] = 0;
    }    
                                    // Base conditions (l==0) :-
  if (window->AmbiguousCharAt(0)) {    // 'N' or 'X' at position 0, alpha[0][bkgIndex] = 1 else 0
    for (i=0; i<_numWM; i++) {
#ifndef _CYCLIC_WINDOWS
      if (i==bkgIndex) _alpha[0][i] = 1;
      else _alpha[0][i] = 0;
#else      
      _alpha[0][i] = _initial[windowindex][i];
#endif
    }
  }
  else {                            // usual case: a known nucleotide at position 0
    for (i=0; i<_numWM; i++) {
      if (_wm_len[i]==1) {
#ifndef _CYCLIC_WINDOWS
	_alpha[0][i] = _pij[bkgIndex][i]*ComputeSequenceProbability(window,0,0,wm[i]);
#else
	DTYPE sum = _initial[windowindex][i]*_pij[i][bkgIndex]*ComputeSequenceProbability(window,0,0,wm[bkgIndex]);
	if (i!=bkgIndex) {
	  for (int j=0; j<_numWM; j++) {
	    sum += _initial[windowindex][j]*_pij[j][i]*ComputeSequenceProbability(window,0,0,wm[i]);
	  }
	}
	_alpha[0][i] = sum;
#endif
      }
      else {
#ifndef _CYCLIC_WINDOWS
	_alpha[0][i] = 0;           // cant have matrix i ending in the first len_i - 1 positions
#else
	                            // but the matrix could have been the initial matrix, and this position is background
	_alpha[0][i] = _initial[windowindex][i]*_pij[i][bkgIndex]*ComputeSequenceProbability(window,0,0,wm[bkgIndex]);
#endif
      }
    }
  }
                                    // Scale the _alpha  
  _c[0] = 0;
  for (i=0; i<_numWM; i++) 
    _c[0] += _alpha[0][i];
  if (_c[0] < SMALL_FLOAT) _c[0] = 1;
  else _c[0] = 1/_c[0];
  _cij[0][0] = _c[0];
  for (i=0; i<_numWM; i++) 
    _alpha[0][i] *= _c[0];
                                    // Recurrences :-
  for (int l=1; l<length; l++) {
    for (int j=0; j<_numWM; j++) {   // compute alpha[l][j]
      if (window->AmbiguousCharAt(l)) {   // 'N' or 'X' here - propagate the previous alpha
	_alpha[l][j] = _alpha[l-1][j];
	continue;
      }
                                    // usual case: known nucleotude at this position
                                    // case 1: position l is background, j ended before l
      DTYPE  sum = _alpha[l-1][j]*_pij[j][bkgIndex]*ComputeSequenceProbability(window,l,l,wm[bkgIndex]);
                                    // case 2: wm_j ends at position l
      if (j!=bkgIndex) {            // since this is different from case 1, background cannot end here
	int wm_len_j = _wm_len[j];
#ifndef _CYCLIC_WINDOWS
	if (l-wm_len_j+1<0)         // case 2 cannot happen                        
	  continue;                 
	if (l-wm_len_j+1==0) {      // case 2 can happen, but there is no alpha for positions before start of wm_j
	  DTYPE  scale_factor = _cij[0][l-1];
	  _alpha[l][j] = scale_factor*_pij[bkgIndex][j]*ComputeSequenceProbability(window,0,l,wm[j]);
	  continue;
	}	
#else
	if (l-wm_len_j+1<0) {       // case 2 cannot happen                        
	  _alpha[l][j] = sum;
	  continue;          
	}       
	if (l-wm_len_j+1==0) {      // case 2 can happen, and there are different possibilities for wm_i before wm_j
	  DTYPE  scale_factor = _cij[0][l-1];
	  for (int i=0; i<_numWM; i++) {
	    sum += scale_factor*_initial[windowindex][i]*_pij[i][j]*ComputeSequenceProbability(window,0,l,wm[j]);
	  }
	  _alpha[l][j] = sum;
	  continue;
	}
#endif
	for (i=0; i<_numWM; i++) {  // case 2 can happen, and there are different possibilities for wm_i before wm_j
	  DTYPE  scale_factor = (wm_len_j>1?_cij[l-wm_len_j+1][wm_len_j-2]:1);
	  sum += _alpha[l-wm_len_j][i]*scale_factor*_pij[i][j]
	    *ComputeSequenceProbability(window,l-wm_len_j+1,l,wm[j]);
	}
      }
      _alpha[l][j] = sum;
    }
                                    // Scale the _alphas
    _c[l] = 0;
    for (int j=0; j<_numWM; j++)
      _c[l] += _alpha[l][j];
    if (_c[l] < SMALL_FLOAT) _c[l] = 1;
    else _c[l] = 1/_c[l];
    for (int j=max(0,l-max_wm_length+1); j<=l-1; j++) {
      _cij[j][l-j] = _cij[j][l-j-1]*_c[l];
    }
    _cij[l][0] = _c[l];
    for (int j=0; j<_numWM; j++) 
      _alpha[l][j] *= _c[l];
  }   
                                    // Clean up
  delete [] wm;
}

void Parameters_H1::Backward()
{
  int i;
  int bkgIndex = BackgroundIndex(); // the background matrix corresponds to last index
  WtMx **wm = new WtMx *[_numWM];
  for (i=0; i<_numWM; i++) {
    if (i!=bkgIndex) {
      wm[i] = _wmc->WM(i);
    }
    else {
      wm[i] = _bkgwm;
    }
  }

  Window *window = _currentWindow;
  int length = window->Length();
                                    // Clear up previous information
  for (int l=0; l<length; l++) {
    for (i=0; i<_numWM; i++) {
      _beta[l][i] = 0;
    }
  }
                                    // Base conditions :-
  if (window->AmbiguousCharAt(length-1)) {   // 'N' or 'X' here
    for (i=0; i<_numWM; i++) {
      _beta[length-1][i] = 1*_c[length-1];
    }
  }
  else {                            // usual case: known nucleotide at this position
    for (i=0; i<_numWM; i++) {
      DTYPE  sum = 0;
      for (int j=0; j<_numWM; j++) {
	sum += _pij[i][j]*ComputeSequenceProbability(window,length-1,length-1,wm[j]);
      }
      _beta[length-1][i] = sum*_c[length-1];
    }
  }
                                    // Recurrences :-
  for (int l=length-2; l>=1; l--) { // need beta only for length-1 .. 1
    for (int j=0; j<_numWM; j++) {  // compute beta[l,j]
      if (window->AmbiguousCharAt(l)) {// 'N' or 'X' here, propagate the previous _beta
	_beta[l][j] = _c[l]*_beta[l+1][j];
	continue;
      }
                                    // usual case: known nucleotide at this position
      DTYPE  sum = 0;
      for (i=0; i<_numWM; i++) {    // any wm_i (including the background) may start at position l
	int wm_len_i = _wm_len[i];
	if (l+wm_len_i<length) {    // wm_i starts and ends before length - 1, so there is a following beta also
	  DTYPE  scale_factor = (wm_len_i>1?_cij[l+1][wm_len_i-2]:1);
	  sum += scale_factor*_pij[j][i]*ComputeSequenceProbability(window,l,l+wm_len_i-1,wm[i])
	    *_beta[l+wm_len_i][i!=bkgIndex?i:j];
	}
	else {                      // wm_i starts here but either it doesnt end, or ends at last position
	  DTYPE  scale_factor = _cij[l+1][length-l-2];
	  sum += scale_factor*_pij[j][i]*ComputeSequenceProbability(window,l,length-1,wm[i]);
	}
      }
      _beta[l][j] = sum*_c[l];
    }
  }
                                    // Clean up
  delete [] wm;
}

void Parameters_H1::PrepareForUpdate()
{
  int i;
  int bkgIndex = BackgroundIndex(); // the background matrix corresponds to last index
  WtMx **wm = new WtMx *[_numWM];
  for (i=0; i<_numWM; i++) {
    if (i!=bkgIndex) {
      wm[i] = _wmc->WM(i);
    }
    else {
      wm[i] = _bkgwm;
    }
  }

  Window *window = _currentWindow;
  int length = window->Length();
  
  int wi = window->GetIndex();
  if ((*_windows)[wi] != window) {
    printf("Error: Couldnt retrieve index of window correctly\n");
    exit(1);
  }
                                    // Clear up previous information
  for (i=0; i<_numWM; i++) {
    for (int j=0; j<_numWM; j++) {
      _Aij[wi][i][j] = -1;
    }
  }
    
    /*********** Note: **********************
    Aijl computed here is actually $\sum_{P|Xijl(P)=1} Pr[P,S|\theta]/(\prod c[l])$,
    which is approximately but not exactly equal to $\sum_{P|Xijl(P)=1} Pr[P,S|\theta]/Pr[S|\theta]$
    This difference does not matter in the computation of the updated probabilities, since the
    correction factor is the same in the numerator and denominator. 
    *****************************************/

  for (i=0; i<_numWM; i++) {
    for (int j=0; j<_numWM; j++) {   // compute _Aij[i][j] ...
      _Aij[wi][i][j] = 0;
      for (int l=0; l<length; l++) {// ... by summing _Aij[i][j][l] over all l
	if (window->AmbiguousCharAt(l)) { // 'N' or 'X' at this position, cant have pij at this position
	  _Aijl_window[i][j][l] = 0;
	  continue;
	}
	int wm_len_j = _wm_len[j];
	if (l==0) {                 // case 1: no alpha term defined for l-1
#ifndef _CYCLIC_WINDOWS
	  if (i==bkgIndex) {        // only possibility at l=0, since the initial state was background
	    _Aijl_window[i][j][l] = _cij[l][wm_len_j-1]*_pij[i][j]
	      *ComputeSequenceProbability(window,l,l+wm_len_j-1,wm[j])
	      *_beta[l+wm_len_j][j!=bkgIndex?j:i];
	    _Aij[wi][i][j] += _Aijl_window[i][j][l];
	  }
	  else _Aijl_window[i][j][l] = 0;
#else
	  if (_initial == NULL) {
	    printf("Error: _initial null for cyclic windows\n");
	    exit(1);
	  }
	  _Aijl_window[i][j][l] =  _cij[l][wm_len_j-1]*_initial[wi][i]*_pij[i][j]
	    *ComputeSequenceProbability(window,l,l+wm_len_j-1,wm[j])
	    *_beta[l+wm_len_j][j!=bkgIndex?j:i];
	  _Aij[wi][i][j] += _Aijl_window[i][j][l];
#endif
	  continue;
	}
	if (l+wm_len_j>=length) {  // no beta term defined for l+wm_len[j]
	  _Aijl_window[i][j][l] = _alpha[l-1][i]*_cij[l][length-l-1]*_pij[i][j]
	    *ComputeSequenceProbability(window,l,length-1,wm[j]);
	  _Aij[wi][i][j] += _Aijl_window[i][j][l];
	  continue;
	}
                                   // general condition: use both alpha and beta
	_Aijl_window[i][j][l] = _alpha[l-1][i]*_cij[l][wm_len_j-1]*_pij[i][j]
	  *ComputeSequenceProbability(window,l,l+wm_len_j-1,wm[j])
	  *_beta[l+wm_len_j][j!=bkgIndex?j:i];
	_Aij[wi][i][j] += _Aijl_window[i][j][l];

      }  
    }
  }

#ifdef _CYCLIC_WINDOWS
  UpdateInitial();
#endif
                               // Clean Up
  delete [] wm;
}

void Parameters_H1::UpdateEmissionProbabilities()
{
#ifdef WARNINGS
  Warn("Warning: UpdateEmissionProbabilities not supported yet\n");
#endif
  return;

#if 0
  int i;
  int bkgIndex = BackgroundIndex(); // the background matrix corresponds to last index
  WtMx **wm = new WtMx *[_numWM];
  for (i=0; i<_numWM; i++) {
    if (i!=bkgIndex) {
      wm[i] = _wmc->WM(i);
    }
    else {
      wm[i] = _bkgwm;
    }
  }

  int numWindows = _windows->size();

  for (i=0; i<_numWM; i++) {
    for (int j=0; j<_wm_len[i]; j++) {
      if (!_free_emission_probabilities[i][j]) continue;
                                    // else recompute emission probabilities
      DTYPE  *Eij = new DTYPE [4];
      for (int k=0; k<4; k++) Eij[k] = 0;
      for (int wi=0; wi<numWindows; wi++) {
	Window *window = (*_windows)[wi];
	int length = window->Length();
	for (int l=0; l<length; l++) {
	  DTYPE  sum = 0;
	  for (int i1=0; i1<_numWM; i1++) {
	    if (l-j < 0) continue;
	    sum += _Aijl[wi][i1][i][l-j];
	  }
	  if (!window->AmbiguousCharAtInReferenceSequence(l))
	    Eij[window->IndexOfCharAtInReferenceSequence(l)] += sum;
	}
      }
      DTYPE  sum = 0;
      for (int k=0; k<4; k++) sum += Eij[k];
      if (sum > 0) {
	for (int k=0; k<4; k++) Eij[k] /= sum;
      }
      wm[i]->UpdateFrequency(j,Eij);
      delete [] Eij;
    }
  }
                                    // Clean Up
  delete [] wm;
#endif 
}

void Parameters_H1::UpdateTransitionProbabilities()
{
  int i;
                                    // save current pij
  for (i=0; i<_numWM; i++) {
    for (int j=0; j<_numWM; j++) {
      _oldpij[i][j] = _pij[i][j];
    }
  }

  int numWindows = _windows->size(); 

  for (i=0; i<_numWM; i++) {
    DTYPE  *aij = new DTYPE [_numWM];
    DTYPE  sum = 0;
    for (int j=0; j<_numWM; j++) {   // compute sum_j _Aijl[i][j]
      aij[j] = 0; 
      for (int wi=0; wi<numWindows; wi++) {
	DTYPE correction = FringeCorrectionFactor(wi);
	aij[j] += _Aij[wi][i][j]*correction;
      }
      sum += aij[j];
    }
    for (int j=0; j<_numWM; j++) {   // update _pij[i][j]
      if (sum > 0) {
	_pij[i][j] = float(aij[j]/sum);
      }
    }
    delete [] aij;
  }
}

void Parameters_H1::Update()
{
  UpdateEmissionProbabilities();
  UpdateTransitionProbabilities();
}

void Parameters_H1::Revert()
{
  int i;

  for (i=0; i<_numWM; i++) {
    for (int j=0; j<_numWM; j++) {
      _pij[i][j] = _oldpij[i][j];
    }
  }

  return;
}

void Parameters_H1::Print(FILE *fp, bool verbose)
{
  int totalLen = 0;
  for (int wi=0; wi<_windows->size(); wi++) {
    totalLen += (*_windows)[wi]->Length();
  }

  if (!verbose) {
    fprintf(fp,"%d\t",(*_windows)[0]->Start());
#ifdef _PRINT_SEQNAME_IN_FENFILE
    char seqname[1024]; (*_windows)[0]->Seq()->Name(seqname);
    fprintf(fp,"%.4f\t%.4f\t%d\t%d\t+\t%s\n",_free_energy_differential,_free_energy_perlength,totalLen,_num_iterations,seqname);
#else 
    fprintf(fp,"%.4f\t%.4f\t%d\t%d\t+\n",_free_energy_differential,_free_energy_perlength,totalLen,_num_iterations);
#endif

    return;
  }
  else {
    for (int wi=0; wi<_windows->size(); wi++)
      (*_windows)[wi]->Print(verbose);
    fprintf(fp,"Score = %.4f\tTotalLen = %d\tIterations = %d\n",_free_energy_differential,totalLen,_num_iterations);
    
    int i;
    
    fprintf(fp,"Transition probabilities (p_ij):\n");   // print the _pij values
    for (i=0; i<_numWM; i++) {                    
      for (int j=0; j<_numWM; j++) {
	fprintf(fp,"%.4f ",_pij[i][j]);
      }
      fprintf(fp,"\n");
    }
  }
}

void Parameters_H1::PrintProbabilities(FILE *fp, bool verbose)
{
  for (int i=0; i<_numWM; i++) {
    for (int j=0; j<_numWM; j++) {
      fprintf(fp,"%.4f ",_pij[i][j]);
    }
    fprintf(fp,"\n");
  }
  return;
}

DTYPE  Parameters_H1::ComputeAverageCount(int i, int j)
{
  DTYPE  totalac = 0;
  int numWindows = _windows->size();
  for (int wi=0; wi<numWindows; wi++) {
    DTYPE  ac = _Aij[wi][i][j];
    DTYPE  correction = FringeCorrectionFactor(wi);
    totalac += ac*correction;
  }

  return totalac;
}

DTYPE  Parameters_H1::ComputeExpectedAverageCount(int i, int j, Parameters_H0* param, DTYPE  *&expectations)
{
#ifdef _CYCLIC_WINDOWS
  Warn("ComputeExpectedAverageCount not supported for cyclic windows\n");
  exit(1);
#endif
  int bkgIndex = BackgroundIndex(); // the background matrix corresponds to last index
  WtMx **wm = new WtMx *[_numWM];
  for (int k=0; k<_numWM; k++) {
    if (k!=bkgIndex) {
      wm[k] = _wmc->WM(k);
    }
    else {
      wm[k] = _bkgwm;
    }
  }

  DTYPE  mean = 0;
  int numWindows = _windows->size();
  expectations = new DTYPE [numWindows];

  for (int wi=0; wi<numWindows; wi++) {
    Window *window = (*_windows)[wi];
    int length = window->NumSpecificCharacters();
    
    DTYPE  *alpha = new DTYPE  [length];
    alpha[0] = 1;
    DTYPE  *bkgpower = new DTYPE [length];
    bkgpower[0] = 1;
    for (int l=1; l<length; l++) {
      alpha[l] = 0;
      for (int k=0; k<_numWM; k++) {
	if (l-_wm_len[k]<0) continue;
	alpha[l] += alpha[l-_wm_len[k]]*param->_pi[k];
      }
      bkgpower[l] = bkgpower[l-1]*param->_pi[bkgIndex];
    }

    DTYPE  factor = 0;
    DTYPE  eac = 0;

    if (i!=bkgIndex) {
      int wm_len_i = _wm_len[i];
      if ((1-bkgpower[1]) < SMALL_FLOAT) {
	for (int l=1; l<=length-wm_len_i; l++) {
	  factor += alpha[l-1]*(length-wm_len_i-l+1);
	}
      }
      else {
	for (int l=1; l<=length-wm_len_i; l++) {
	  factor += alpha[l-1]*(1-bkgpower[length-wm_len_i-l+1]);
	}
	factor *= 1/(1-bkgpower[1]);
      }
      eac = param->_pi[i]*param->_pi[j]*factor;
    }
    
    if (i==bkgIndex && j!=bkgIndex) {
      for (int l=0; l<length; l++) {
	factor += bkgpower[l];
      }
      eac = param->_pi[j]*factor;
    }
    
    if (i==bkgIndex && j==bkgIndex) {
      for (int l=0; l<length; l++) {
	factor += bkgpower[l];
      }
      eac = param->_pi[bkgIndex]*factor;
    }
    expectations[wi] = eac;    
    mean += eac;

    delete [] bkgpower;
    delete [] alpha;
  }

  delete [] wm;
  
  return mean;
}

DTYPE   Parameters_H1::ComputeVarianceOfCount(int i, int j, Parameters_H0* param, DTYPE  *expectations)
{
#ifdef _CYCLIC_WINDOWS
  Warn("ComputeVarianceOfCount not supported for cyclic windows\n");
  exit(1);
#endif
  int bkgIndex = BackgroundIndex();  // the background matrix corresponds to last index
  WtMx **wm = new WtMx *[_numWM];
  for (int k=0; k<_numWM; k++) {
    if (k!=bkgIndex) {
      wm[k] = _wmc->WM(k);
    }
    else {
      wm[k] = _bkgwm;
    }
  }

  DTYPE  variance = 0;
  int numWindows = _windows->size();
  for (int wi=0; wi<numWindows; wi++) {
    Window *window = (*_windows)[wi];
    int length = window->NumSpecificCharacters();
 
    DTYPE  *alpha = new DTYPE  [length];
    alpha[0] = 1;
    int c = 0;
    DTYPE  *bkgsqrpower = new DTYPE [length];
    DTYPE  *bkgpower = new DTYPE [length];
    bkgsqrpower[0] = 1;
    bkgpower[0] = 1;
    for (int l=1; l<length; l++) {
      alpha[l] = 0;
      for (int k=0; k<_numWM; k++) {
	if (l-_wm_len[k]<0) continue;
	alpha[l] += alpha[l-_wm_len[k]]*param->_pi[k];
      }
      bkgpower[l] = bkgpower[l-1]*param->_pi[bkgIndex];
      bkgsqrpower[l] = bkgsqrpower[l-1] + (l+1)*bkgpower[l];
      // detect if alpha has saturated
      DTYPE  change = abs((alpha[l]-alpha[c])/alpha[l]);
      if (change > ALPHATHRESHOLD) {
	c = l;
      }
    }
    if (c < 1) c = 1;
    // So S[x] = bkgsqrpower[L'-x]

    // also compute the average alpha_c to be used as the constant for alpha[c..length-1]
    DTYPE  alpha_c = 0;
    for (int l=c; l<length; l++) {
      alpha_c += alpha[l];
    }
    if (length-c > 0) alpha_c /= float(length-c);
    
    
    DTYPE  factor = 0;
    
    if (i!=bkgIndex && j!=bkgIndex && i!=j) {
      int Lprime = length-1-2*_wm_len[i]-_wm_len[j];
      if (c>=Lprime) c = Lprime;
      
      // the B term first
      DTYPE  B = 0;
      for (int x1plusx2=2*c; x1plusx2<=Lprime; x1plusx2++) {
	B += (x1plusx2-2*c+1)*bkgsqrpower[Lprime-x1plusx2];
      }     
      B *= pow(alpha_c,2);
      factor += B;
      
      // then the A term
      DTYPE  A = 0;
      DTYPE  *SS = new DTYPE [c];
      SS[c-1] = 0;
      for (int m=2*c-1; m<=Lprime; m++) SS[c-1] += bkgsqrpower[Lprime-m];
      for (int l=c-2; l>=0; l--) {
	SS[l] = SS[l+1];
	if (c+l<=Lprime) 
	  SS[l] += bkgsqrpower[Lprime-(c+l)];
      }
      for (int x1=0; x1<c; x1++) {
	A += alpha[x1]*SS[x1];
      }
      A *= alpha_c;
      delete [] SS;
      A *= 2;   // the symmetric case is added too
      
      for (int x1=0; x1<c; x1++) {
	for (int x2=0; x2<c; x2++) {
	  if (x1+x2 > Lprime) continue;
	  A += alpha[x2]*alpha[x1]*bkgsqrpower[Lprime-x1-x2];
	}
      }
      factor += A;
      
      // then scale the factor
      factor *= 2*pow((param->_pi[i]*param->_pi[j]),2);
    }
    if (i!=bkgIndex && j!=bkgIndex && i==j) {
      int Lprime = length-1-3*_wm_len[i];
      DTYPE  factor1 = 0;
      if (c>=Lprime) c = Lprime;
      
      // the B term first
      DTYPE  B = 0;
      for (int x1plusx2=2*c; x1plusx2<=Lprime; x1plusx2++) {
	B += (x1plusx2-2*c+1)*bkgsqrpower[Lprime-x1plusx2];
      }     
      B *= pow(alpha_c,2);
      factor1 += B;
      
      // then the A term
      DTYPE  A = 0;
      DTYPE  *SS = new DTYPE [c];
      SS[c-1] = 0;
      for (int m=2*c-1; m<=Lprime; m++) SS[c-1] += bkgsqrpower[Lprime-m];
      for (int l=c-2; l>=0; l--) {
	SS[l] = SS[l+1];
	if (c+l<=Lprime) 
	  SS[l] += bkgsqrpower[Lprime-(c+l)];
      }
      for (int x1=0; x1<c; x1++) {
	A += alpha[x1]*SS[x1];
      }
      A *= alpha_c;
      delete [] SS;
      A *= 2;   // the symmetric case is added too
      
      for (int x1=0; x1<c; x1++) {
	for (int x2=0; x2<c; x2++) {
	  if (x1+x2 > Lprime) continue;
	  A += alpha[x2]*alpha[x1]*bkgsqrpower[Lprime-x1-x2];
	}
      }
      factor1 += A;
      
      // then scale the factor
      factor1 *= 2*pow(param->_pi[i],4);
      
      int Lprimeprime = length-1-2*_wm_len[i];
      DTYPE  factor2 = 0; // additional possibilities in this case
      for (int x1=0; x1<=Lprimeprime; x1++) {
	factor2 += alpha[x1]*bkgsqrpower[Lprimeprime-x1];
      }
      factor2 *= 2*pow(param->_pi[i],3);
      factor = factor1+factor2;
    }
    if (i!=bkgIndex && j==bkgIndex) {
      int Lprime = length-1-2*_wm_len[i]-_wm_len[j];
      DTYPE  factor1 = 0;
      if (c>=Lprime) c = Lprime;
      // the B term first
      DTYPE  B = 0;
      for (int x1plusx2=2*c; x1plusx2<=Lprime; x1plusx2++) {
	B += (x1plusx2-2*c+1)*bkgsqrpower[Lprime-x1plusx2];
      }     
      B *= pow(alpha_c,2);
      factor1 += B;
      
      // then the A term
      DTYPE  A = 0;
      DTYPE  *SS = new DTYPE [c];
      SS[c-1] = 0;
      for (int m=2*c-1; m<=Lprime; m++) SS[c-1] += bkgsqrpower[Lprime-m];
      for (int l=c-2; l>=0; l--) {
	SS[l] = SS[l+1];
	if (c+l<=Lprime) 
	  SS[l] += bkgsqrpower[Lprime-(c+l)];
      }
      for (int x1=0; x1<c; x1++) {
	A += alpha[x1]*SS[x1];
      }
      A *= alpha_c;
      delete [] SS;
      A *= 2;   // the symmetric case is added too
      
      for (int x1=0; x1<c; x1++) {
	for (int x2=0; x2<c; x2++) {
	  if (x1+x2 > Lprime) continue;
	  A += alpha[x2]*alpha[x1]*bkgsqrpower[Lprime-x1-x2];
	}
      }
      factor1 += A;
      
      // then scale the factor
      factor1 *= 2*pow((param->_pi[i]*param->_pi[j]),2);
      
      int Lprimeprime = length-1-_wm_len[i]-_wm_len[j];
      DTYPE  factor2 = 0; // additional possibilities in this case
      for (int x1=0; x1<=Lprimeprime; x1++) {
	factor2 += alpha[x1]*bkgsqrpower[Lprimeprime-x1];
      }
      factor2 *= 2*param->_pi[i]*pow(param->_pi[j],2);
      factor = factor1+factor2;
    }
    if (i==bkgIndex && j!=bkgIndex) {
      factor = 0; 
    }
    if (i==bkgIndex && j==bkgIndex) {
      int Lprime = length-1-_wm_len[i];
      factor = 2*pow(param->_pi[i],2)*bkgsqrpower[Lprime-0];
    }
    
    variance += (expectations[wi] + factor - pow(expectations[wi],2));
    
    delete [] bkgpower;
    delete [] bkgsqrpower;
    delete [] alpha;
  }

  delete [] wm;  
  
  if (variance < -1) {
#ifdef WARNINGS
    char warning[10000];
    sprintf(warning,"Warning: Negative variance found: %d %d %g\n",i,j,variance); 
    Warn(warning);
#endif
  }
  if (variance < 0) variance = 0;
  return variance;
}

void Parameters_H1::PrintAverageCounts(FILE *fp, bool verbose)
{
  int bkgIndex = BackgroundIndex(); // the background matrix corresponds to last index
  WtMx **wm = new WtMx *[_numWM];
  for (int i=0; i<_numWM; i++) {
    if (i!=bkgIndex) {
      wm[i] = _wmc->WM(i);
    }
    else {
      wm[i] = _bkgwm;
    }
  }
  
  int numWindows = _windows->size();
  
  for (int i=0; i<_numWM; i++) {
    DTYPE  totalaverage = 0;
    int numWindows = _windows->size();
    for (int wi=0; wi<numWindows; wi++) {
      DTYPE  correction = FringeCorrectionFactor(wi);
      DTYPE  average = 0;
      for (int j=0; j<_numWM; j++) average += _Aij[wi][j][i];
      average *= correction;
      totalaverage += average;
    }
    fprintf(fp,"%.1f ",totalaverage);
  }
  fprintf(fp,"\n");
  delete [] wm;

  return;
}

#include <list>
struct tmpstruc1 {
  int mindex;
  int offset;
  DTYPE ail;
};

void Parameters_H1::PrintProfile(FILE *fp, FILE *dict, float occurrence_threshold)
{
  int bkgIndex = BackgroundIndex(); // the background matrix corresponds to last index
  WtMx **wm = new WtMx *[_numWM];
  for (int i=0; i<_numWM; i++) {
    if (i!=bkgIndex) {
      wm[i] = _wmc->WM(i);
    }
    else {
      wm[i] = _bkgwm;
    }
  }
  
  int numWindows = _windows->size();
  for (int wi=0; wi<numWindows; wi++) {
    Window *window = (*_windows)[wi];
    char name[1024];
    window->Seq()->Name(name);
    fprintf(fp,">\n");
    fprintf(fp,"Sequence : %s\tPosition %d\t+\n",name,window->Start());
    fprintf(dict,">%s\nPosition: %d\tNucl: %d\tWord_av_length: 0.00\tFree Energy: %.4f\t+\n",name,window->Start(),window->NumSpecificCharacters(),_free_energy_differential);
    
    _currentWindow = window;
    Forward();
    Backward();
    PrepareForUpdate();

    int length = window->Length();
    DTYPE  correction = FringeCorrectionFactor(wi);      	
    list<tmpstruc1 *> worklist;

    for (int l=0; l<length; l++) {
      fprintf(fp,"%d\t%c\n",window->Start()+l,window->Seq()->CharAt(window->Start()+l));
      if (window->AmbiguousCharAt(l)) {
	fprintf(fp,"N\t\t\t\t1.0\n");
	continue;
      }

      // process the working list
      for (list<tmpstruc1 *>::iterator it = worklist.begin(); it != worklist.end();) {
	tmpstruc1 *&current = *it;
	char name[1024]; wm[current->mindex]->Name(name,15);
	if (current->offset <= wm[current->mindex]->Length()) {
	  fprintf(fp,"%s\t+\t%d\t%.4f\n",name,current->offset,current->ail);
	}
	current->offset ++;
	if (current->offset > wm[current->mindex]->Length()) {
	  it = worklist.erase(it);
	  delete current;
	}
	else it++;
      }      

      // see if there's a new motif starting here
      for (int i=0; i<_numWM; i++) {
	if (l >= length-_wm_len[i]+1) continue;
	DTYPE  ail = 0;
	for (int j=0; j<_numWM; j++) 
	  ail += _Aijl_window[j][i][l]*correction;
	if (ail > occurrence_threshold) {
	  char name[1024]; wm[i]->Name(name,15);
	  if (i==bkgIndex) {
	    fprintf(fp,"%s\t\t\t%.4f\n",name,ail);
	  }
	  else {
	    fprintf(fp,"%s\t+\t1\t%.4f\n",name,ail);
	    // also put it in a list
	    tmpstruc1 *tmp = new tmpstruc1;
	    tmp->mindex = i;
	    tmp->offset = 2;
	    tmp->ail    = ail;
	    worklist.push_back(tmp);
	  }
	}
      }      
    }

    // delete the working list
    for (list<tmpstruc1 *>::iterator it = worklist.begin(); it != worklist.end(); ) {
      tmpstruc1 *&current = *it;
      it = worklist.erase(it);
      delete current;
    }          

    // write the dictionary
    for (int i=0; i<_numWM; i++) {
      DTYPE average = 0;
      for (int j=0; j<_numWM; j++) average += _Aij[wi][j][i];
      average *= correction;
      if (average > occurrence_threshold) {
	char name[1024]; wm[i]->Name(name,15);
	fprintf(dict,"%s\t0.000000\t%.4f\n",name,average);
      }
    }

    fprintf(dict,"<\n");
    fprintf(fp,"<\n");
  }   

  delete [] wm;
  return;
}

#ifdef _CYCLIC_WINDOWS

void Parameters_H1::UpdateInitial()
{
  int i;
  int bkgIndex = BackgroundIndex(); // the background matrix corresponds to last index
  WtMx **wm = new WtMx *[_numWM];
  for (i=0; i<_numWM; i++) {
    if (i!=bkgIndex) {
      wm[i] = _wmc->WM(i);
    }
    else {
      wm[i] = _bkgwm;
    }
  }
  
  Window *window = _currentWindow;
  int length = window->Length();
  
  int wi = window->GetIndex();
  if ((*_windows)[wi] != window) {
    printf("Error: Couldnt retrieve index of window correctly\n");
    exit(1);
  }
  for (i=0; i<_numWM; i++) {
    _initial[wi][i] = 0;
  }

  DTYPE  term = FringeCorrectionFactorHelper(window);
  DTYPE  correction = 1/term;  	
  list<tmpstruc1 *> worklist;

  int last_motif_starts_at = -1;
  float remaining_probability = 1;
  for (int l=length-1; l>=0; l--) {
    if (window->AmbiguousCharAt(l)) {
      continue;
    }
    
    // see if there's a new motif starting here
    for (i=0; i<_numWM; i++) {
      if (i==bkgIndex) continue;
      if (l >= length-_wm_len[i]+1) continue;
      DTYPE ail = 0;
      for (int j=0; j<_numWM; j++) {
	ail += _Aijl_window[j][i][l]*correction;
      }
      if (ail < SMALL_MOTIF_OCCURRENCE_THRESHOLD) continue;
      tmpstruc1 *tmp = new tmpstruc1;
      tmp->mindex = i;
      tmp->offset = l;
      tmp->ail    = ail*remaining_probability;
      worklist.push_back(tmp);
      remaining_probability -= tmp->ail;
    }

    if (remaining_probability < SMALL_FLOAT) break;
  }

  // process the working list
  int nummotifsfound = 0;
  for (list<tmpstruc1 *>::iterator it = worklist.begin(); it != worklist.end();) {
    tmpstruc1 *&current = *it;
    _initial[wi][current->mindex] += current->ail;
    nummotifsfound++;
    it++;
  }
  float sum = 0;
  for (int j=0; j<_numWM; j++) sum += _initial[wi][j];
  if (sum < 1) _initial[wi][bkgIndex] = 1-sum;
  else {
    for (int j=0; j<_numWM; j++) _initial[wi][j] /= sum;
  }
  
  // delete the working list
  for (list<tmpstruc1 *>::iterator it = worklist.begin(); it != worklist.end();) {
    tmpstruc1 *&current = *it;
    it = worklist.erase(it);
    delete current;
  }

  return;
}
#endif


Parameters_H1::Parameters_H1()
{
  _pij = _oldpij = NULL;
  _alpha = _beta =_cij = NULL;
  _Aij = NULL;
  _Aijl_window = NULL;
  _c = NULL; 
}

Parameters_H1::~Parameters_H1()
{
  Destroy();
}

void Parameters_H1::Destroy()
{
  int i;

  if (_pij) {
    for (i=0; i<_numWM; i++) {
      if (_pij[i]!=NULL) delete [] _pij[i];
    }
    delete [] _pij;
  }

  if (_oldpij) {
    for (i=0; i<_numWM; i++) {
      if (_oldpij[i]!=NULL) delete [] _oldpij[i];
    }
    delete [] _oldpij;
  }


  if (_free_emission_probabilities) {
    for (i=0; i<_numWM; i++) {
      delete [] _free_emission_probabilities[i];
    }
    delete [] _free_emission_probabilities;
  }

  if (_bkgwm) delete _bkgwm;
  if (_wm_len) delete [] _wm_len;

  if (_windows==NULL) return;

#ifdef _CYCLIC_WINDOWS
  if (_initial) {
    int numWindows = _windows->size();
    for (int wi=0; wi<numWindows; wi++) {
      delete [] _initial[wi];
    }
    delete [] _initial;
    _initial = NULL;
  }
#endif

  if (_Aij) {
    int numWindows = _windows->size();
    for (int wi=0; wi<numWindows; wi++) {
      if (_Aij[wi]) {
	for (i=0; i<_numWM; i++) {
	  if (_Aij[wi][i]) {
	    delete [] _Aij[wi][i];
	  }
	}
	delete [] _Aij[wi];
      }
    }

    delete [] _Aij;
    _Aij = NULL;
  }

  if (_Aijl_window) {
    for (i=0; i<_numWM; i++) {
      if (_Aijl_window[i]) {
	for (int j=0; j<_numWM; j++) {
	  if (_Aijl_window[i][j]) {
	    delete [] _Aijl_window[i][j];
	  }
	}
	delete [] _Aijl_window[i];
      }
    }
    delete [] _Aijl_window;
    _Aijl_window = NULL;
  }

  if (_alpha) {
    for (int l=0; l<_max_window_length; l++) delete [] _alpha[l];
    delete [] _alpha;
    _alpha = NULL;
  }

  if (_beta) {
    for (int l=0; l<_max_window_length; l++) delete [] _beta[l];
    delete [] _beta;
    _beta = NULL;
  }

  if (_c) {
    delete [] _c;
    _c = NULL;
  }

  if (_cij) {
    for (int l=0; l<_max_window_length; l++) delete [] _cij[l];
    delete [] _cij;
    _cij = NULL;
  }

  if (_fringe_corrections) {
    delete [] _fringe_corrections;
    _fringe_corrections = NULL;
  }

  // destroy the cache if present
  if (_currentwindowcache) {
    int numWindows = _windows->size();
    for (int wi=0; wi<numWindows; wi++) _currentwindowcache[wi].Destroy();
    delete [] _currentwindowcache;
    _currentwindowcache = NULL;
  }

  return;
}

Parameters_H1::Parameters_H1(const Parameters_H1 &p)
{
  Copy(p);
}

Parameters_H1& Parameters_H1::operator=(const Parameters_H1 &p)
{
  if (this == &p) return *this;
  Destroy();
  Copy(p);
}

void Parameters_H1::Copy(const Parameters_H1 &p)
{
  _wmc = p._wmc;
  _numWM = p._numWM;
  _windows = p._windows;
  _is_initialized = p._is_initialized;
  _is_trained = p._is_trained;

  int i;

  if (_numWM && p._pij) {
    _pij = new float *[_numWM];
    for (i=0; i< _numWM; i++) {
      _pij[i] = new float[_numWM];
      for (int j=0; j<_numWM; j++) {
	_pij[i][j] = p._pij[i][j];
      }
    }
  }
  else _pij = NULL;

  if (_numWM && p._oldpij) {
    _oldpij = new float *[_numWM];
    for (i=0; i< _numWM; i++) {
      _oldpij[i] = new float[_numWM];
      for (int j=0; j<_numWM; j++) {
	_oldpij[i][j] = p._oldpij[i][j];
      }
    }
  }
  else _oldpij = NULL;

  int bkgIndex = BackgroundIndex();
  if (_numWM && p._free_emission_probabilities) {
    _free_emission_probabilities = new bool *[_numWM];
    WtMx *w;
    for (i=0; i<_numWM; i++) {
      if (i==bkgIndex) w = p._bkgwm;
      else w = p._wmc->WM(i);
      _free_emission_probabilities[i] = new bool[w->Length()];
      for (int j=0; j<w->Length(); j++) {
	_free_emission_probabilities[i][j] = p._free_emission_probabilities[i][j];
      }
    }
  }
  else _free_emission_probabilities = NULL;

  if (_windows && _numWM && p._Aij) {
    int numWindows = _windows->size();
    _Aij = new DTYPE  **[numWindows];
    for (int wi=0; wi<numWindows; wi++) {
      Window *window = (*_windows)[wi];
      int length = (*_windows)[wi]->Length();
      _Aij[wi] = new DTYPE  *[_numWM];
      for (i=0; i< _numWM; i++) {
	_Aij[wi][i] = new DTYPE [_numWM];
	for (int j=0; j<_numWM; j++) {
	  _Aij[wi][i][j] = p._Aij[wi][i][j];
	}
      }
    }
  }
  else _Aij = NULL;

#ifdef _CYCLIC_WINDOWS
  if (p._initial) {
    int numWindows = _windows->size();
    _initial = new float *[numWindows];
    for (int wi=0; wi<numWindows; wi++) {
      _initial[wi] = new float[_numWM];
      for (int j=0; j<_numWM; j++) 
	_initial[wi][j] = p._initial[wi][j];
    }
  }
#endif

  _currentWindow = NULL;
  _alpha = _beta =_cij = NULL;
  _c = NULL;
  _Aijl_window = NULL;

  _bkgwm = new WtMx(p._bkgwm);
  _free_energy = p._free_energy;
  _free_energy_differential = p._free_energy_differential;
  _initialbias = p._initialbias;
  _num_iterations = p._num_iterations;
  _is_initialized = p._is_initialized;

  return;
}

Parameters_H1::Parameters_H1(WtMxCollection *wmc, int extreme)
{
  int i;

  _wmc = wmc;
  _numWM = _wmc->Size()+1;
  _pij = new float *[_numWM];
  _oldpij = new float *[_numWM];
  for (i=0; i<_numWM; i++) {
    _pij[i] = new float[_numWM];
    _oldpij[i] = new float[_numWM];
    for (int j=0; j<_numWM; j++) {
      if (j==extreme) {
	_pij[i][j] = ALMOST_ONE;
      }
      else {
	_pij[i][j] = (1-ALMOST_ONE)/(_numWM-1);
      }
      _oldpij[i][j] = _pij[i][j];
    }
  }

  _currentWindow = NULL;
  _alpha = _beta =_cij = NULL;
  _Aij = NULL;
  _Aijl_window = NULL;
  _bkgwm = NULL;
  _c = NULL; 
  _free_energy = 0;
  _free_energy_differential = 0;
#ifdef _CYCLIC_WINDOWS
  _initial = NULL;
#endif
  _is_initialized = true;
}


DTYPE  Parameters_H1::Norm_of_parameter_difference()
{
  DTYPE  sum = 0;
  for (int i=0; i<_numWM; i++) {
    for (int j=0; j<_numWM; j++) {
      sum += (_pij[i][j]-_oldpij[i][j])*(_pij[i][j]-_oldpij[i][j]);
    }
  }
  return sqrt(sum)/(_numWM*_numWM);
}

char *Parameters_H1::CreateSequence(int length, int randseed, bool verbose)
{
  int i;
  char *sequence = new char[length+1];
  int bkgIndex = BackgroundIndex(); // the background matrix corresponds to last index
  WtMx **wm = new WtMx *[_numWM];
  for (i=0; i<_numWM; i++) {
    if (i!=bkgIndex) {
      wm[i] = _wmc->WM(i);
    }
    else {
      wm[i] = _bkgwm;
    }
  }
                                    
  float **cum = new float*[_numWM];
  for (i=0; i<_numWM; i++) {
    cum[i] = new float[_numWM];
    cum[i][0] = _pij[i][0];
    for (int j=1; j<_numWM; j++) {
      cum[i][j] = cum[i][j-1] + _pij[i][j];
    }
  }

  int *count = new int[_numWM];
  for (i=0; i<_numWM; i++) count[i] = 0;
  int **count_ij = new int *[_numWM];
  for (i=0; i<_numWM; i++) {
    count_ij[i] = new int[_numWM];
    for (int j=0; j<_numWM; j++)
      count_ij[i][j] = 0;
  }

  srandom(randseed);
  int pos = 0;
  int lastmotif = bkgIndex;
  while (pos < length) {
    float r = random()/float(RAND_MAX);
    int rindex = -1;
    if (r < cum[lastmotif][0]) {
      rindex = 0; 
    }
    else {
      for (int j=1; j<_numWM; j++) {
	if (r < cum[lastmotif][j]) {
	  rindex = j;
	  break;
	}
      }
      if (rindex==-1) rindex = _numWM-1;
    }
    // plant a wm[rindex]
    // but first decide orientation:
    int orientation = count[rindex]%2;
    for (int l=0; l<_wm_len[rindex]; l++) {
      if (orientation==0) 
	sequence[pos] = wm[rindex]->GetRandomChar(l);
      else 
	sequence[pos] = ReverseChar(wm[rindex]->GetRandomChar(_wm_len[rindex]-1-l));
      pos++;
      if (pos >= length) break;
    }
    count[rindex]++;
    count_ij[lastmotif][rindex]++;
    if (rindex != bkgIndex) 
      lastmotif = rindex;
  }

  if (verbose) {
    printf("Planting stats: ");
    for (i=0; i<_numWM; i++) printf("%d %d, ",i,count[i]);
    printf("\n");
    for (i=0; i<_numWM; i++) {
      for (int j=0; j<_numWM; j++) 
	printf("%d ",count_ij[i][j]);
      printf("\n");
    }      
  }

  sequence[length] = 0;

  delete [] count;
  delete [] cum;
  delete [] wm;

  return sequence;
}

void Parameters_H1::ForceCorrelation(int l, int r, float strength)
{
  // change _pij[l][r]
  // modify all other _pij[l][*] accordingly
  if ( _pij[l][r]*(1+strength)>=1) {
    _pij[l][r] = 1;
    for (int j=0; j<_numWM; j++) {
      if (j!=r) _pij[l][j] = 0;
    }
    return;
  }
  _pij[l][r] = _pij[l][r]*(1+strength);
  DTYPE  sumj = 0;
  for (int j=0; j<_numWM; j++) {
    if (j!=r) sumj += _pij[l][j];
  }
  for (int j=0; j<_numWM; j++) {
    if (j!=r) _pij[l][j] = (_pij[l][j]/sumj)*(1-_pij[l][r]);
  }
  return;
}
